// MVTools by Manao
// General classe for motion based filters

// See legal notice in Copying.txt for more information

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA, or visit
// http://www.gnu.org/copyleft/gpl.html .

#include "MVAnalyse.h"
#include <stdio.h>
#include <math.h>


extern MVCore mvCore;

MVAnalyse::MVAnalyse(PClip _child, int size, int pel, int lv, int st, int stp,
                     bool isb, int lambda, bool chroma, int df, bool _mmx, bool _isse, int _idx,
                     int _lsad, int _pnew, int _plevel, bool _global, int _overlap, IScriptEnvironment* env) :
GenericVideoFilter(_child),
analysisData()
{
	analysisData.nWidth = vi.width;
//	if ( analysisData.nWidth & 7 ) env->ThrowError("MVAnalyse: Width has to be dividable by 8");

//	if ( (analysisData.nWidth & 15) && size==16) env->ThrowError("MVAnalyse: Width has to be dividable by 16 for block size=16");

	analysisData.nHeight = vi.height;
//	if ( analysisData.nHeight & 7 ) env->ThrowError("MVAnalyse: Height has to be dividable by 8");

//	if ( (analysisData.nHeight & 15) && size==16) env->ThrowError("MVAnalyse: Height has to be dividable by 16 for block size=16");

	analysisData.nBlkSize = size;
	if (( analysisData.nBlkSize != 4 ) && ( analysisData.nBlkSize != 8 ) && ( analysisData.nBlkSize != 16 ))
		env->ThrowError("MVAnalyse: Block's size must be 4, 8 or 16");

	if (!vi.IsYV12() && !vi.IsYUY2())
		env->ThrowError("MVAnalyse: Clip must be YV12 or YUY2");

//	if (vi.IsYUY2() && (size != 8 && size != 16) )
//		env->ThrowError("MVAnalyse: Block size must be 8 or 16 for YUY2");

   analysisData.nPel = pel;
	if (( analysisData.nPel != 1 ) && ( analysisData.nPel != 2 ))
		env->ThrowError("MVAnalyse: pel has to be 1 or 2");

   analysisData.nDeltaFrame = df;
   if ( analysisData.nDeltaFrame < 1 )
      analysisData.nDeltaFrame = 1;
   if ( analysisData.nDeltaFrame > 8 )
      analysisData.nDeltaFrame = 8;

   if (_overlap<0 || _overlap >= size)
		env->ThrowError("MVAnalyse: overlap must be less than block size");

   if (_overlap%2)
		env->ThrowError("MVAnalyse: overlap must be even");

	analysisData.nOverlap = _overlap; 

	int nBlkX = (analysisData.nWidth - analysisData.nOverlap) / (analysisData.nBlkSize - analysisData.nOverlap);
//	if (analysisData.nWidth > (analysisData.nBlkSize - analysisData.nOverlap) * nBlkX ) 
//		nBlkX += 1;
		
	int nBlkY = (analysisData.nHeight - analysisData.nOverlap) / (analysisData.nBlkSize - analysisData.nOverlap);
//	if (analysisData.nHeight > (analysisData.nBlkSize - analysisData.nOverlap) * nBlkY ) 
//		nBlkY += 1;

	analysisData.nBlkX = nBlkX; 
	analysisData.nBlkY = nBlkY; 
	int nWidth_B = nBlkX*(analysisData.nBlkSize - analysisData.nOverlap) + analysisData.nOverlap;
	int nHeight_B = nBlkY*(analysisData.nBlkSize - analysisData.nOverlap) + analysisData.nOverlap;

	analysisData.nLvCount = ilog2(	((analysisData.nWidth) > (analysisData.nHeight)) ? (nBlkY) : (nBlkX) ) - lv;
	analysisData.nLvCount = ( analysisData.nLvCount < 1 ) ? 1 : analysisData.nLvCount;

	analysisData.isBackward = isb;


   nLambda = lambda;
   lsad = _lsad;
   pnew = _pnew;
   plevel = _plevel;
   global = _global;

   //  we'll transmit to the processing filters a handle
   // on the analyzing filter itself ( it's own pointer ), in order
   // to activate the right parameters.
   vi.nchannels = reinterpret_cast<int>(&analysisData);

   mmx = _mmx && ( env->GetCPUFlags() & CPUF_MMX );

	isse = _isse && ( env->GetCPUFlags() & CPUF_INTEGER_SSE );

	switch ( st )
	{
	case 0 : 
		searchType = ONETIME;
		nSearchParam = ( stp < 1 ) ? 1 : stp;
		break;
	case 1 :
		searchType = NSTEP;
		nSearchParam = ( stp < 0 ) ? 0 : stp;
		break;
	case 2 : 
		searchType = LOGARITHMIC;
		nSearchParam = ( stp < 1 ) ? 1 : stp;
		break;
	case 3 : 
		searchType = EXHAUSTIVE;
		nSearchParam = ( stp < 0 ) ? 0 : stp;
		break;
	default :
		searchType = LOGARITHMIC;
		nSearchParam = ( stp < 1 ) ? 1 : stp;
		break;
	}

	if (!vi.IsYV12() && !vi.IsYUY2())
		env->ThrowError("MVAnalyse: Needs YV12 or YUY2 data");
	analysisData.pixelType = vi.pixel_type;
	analysisData.yRatioUV = (vi.IsYV12()) ? 2 : 1;

   analysisData.nFlags = 0;
   analysisData.nFlags |= _isse ? MOTION_USE_ISSE : 0;
   analysisData.nFlags |= _mmx ? MOTION_USE_MMX : 0;
   analysisData.nFlags |= analysisData.isBackward ? MOTION_IS_BACKWARD : 0;
   analysisData.nFlags |= chroma ? MOTION_USE_CHROMA_MOTION : 0;

	vectorFields = new GroupOfPlanes(analysisData.nWidth, analysisData.nHeight, analysisData.nBlkSize, 
		analysisData.nLvCount, analysisData.nPel, analysisData.nFlags, 
		analysisData.nOverlap, analysisData.nBlkX, analysisData.nBlkY, analysisData.yRatioUV);

	vi.height = 1;
	vi.width = vectorFields->GetArraySize();
	vi.pixel_type = VideoInfo::CS_BGR32;

   analysisData.nIdx = _idx;

   analysisData.nMagicKey = MOTION_MAGIC_KEY;

   analysisData.pmvCore = &mvCore;

   int hpad = analysisData.nBlkSize; 
   int vpad = analysisData.nBlkSize; 
   mvCore.AddFrames(analysisData.nIdx, MV_BUFFER_FRAMES, analysisData.nLvCount, 
	   analysisData.nWidth, analysisData.nHeight, analysisData.nPel, hpad, 
	   vpad, YUVPLANES, _isse, analysisData.yRatioUV);

   if ( (analysisData.pixelType & VideoInfo::CS_YUY2) == VideoInfo::CS_YUY2 )
   {
		SrcPlanes =  new YUY2Planes(analysisData.nWidth, analysisData.nHeight);
		RefPlanes =  new YUY2Planes(analysisData.nWidth, analysisData.nHeight);
   }
}

MVAnalyse::~MVAnalyse()
{
   if ( (analysisData.pixelType & VideoInfo::CS_YUY2) == VideoInfo::CS_YUY2 )
   {
	delete SrcPlanes;
	delete RefPlanes;
   }
	delete vectorFields;
}

PVideoFrame __stdcall MVAnalyse::GetFrame(int n, IScriptEnvironment* env)
{
	PVideoFrame	src = child->GetFrame(n, env);
	PVideoFrame	dst = env->NewVideoFrame(vi);
	unsigned char *pDst = dst->GetWritePtr();
	const unsigned char *pSrc, *pSrcU, *pSrcV;
	const unsigned char *pRef, *pRefU, *pRefV;
	int nSrcPitch, nSrcPitchUV;
	int nRefPitch, nRefPitchUV;

	int minframe = ( analysisData.isBackward ) ? 0 : analysisData.nDeltaFrame;
	int maxframe = ( analysisData.isBackward ) ? vi.num_frames - analysisData.nDeltaFrame : vi.num_frames;
	int offset = ( analysisData.isBackward ) ? analysisData.nDeltaFrame : -analysisData.nDeltaFrame;

	if ( (analysisData.pixelType & VideoInfo::CS_YUY2) == VideoInfo::CS_YUY2 )
	{
		const unsigned char *pSrcYUY2 = src->GetReadPtr();
		int nSrcPitchYUY2 = src->GetPitch();
		pSrc = SrcPlanes->GetPtr();
		pSrcU = SrcPlanes->GetPtrU();
		pSrcV = SrcPlanes->GetPtrV();
		nSrcPitch  = SrcPlanes->GetPitch();
		nSrcPitchUV  = SrcPlanes->GetPitchUV();
		YUY2ToPlanes(pSrcYUY2, nSrcPitchYUY2, analysisData.nWidth, analysisData.nHeight, 
			pSrc, nSrcPitch, pSrcU, pSrcV, nSrcPitchUV, isse);
	}
	else
	{
		pSrc = src->GetReadPtr(PLANAR_Y);
	    pSrcU = src->GetReadPtr(PLANAR_U);
	    pSrcV = src->GetReadPtr(PLANAR_V);
		nSrcPitch = src->GetPitch(PLANAR_Y);
	    nSrcPitchUV = src->GetPitch(PLANAR_U);
	}

	if (( n < maxframe ) && ( n >= minframe ))
	{
		if ( (analysisData.pixelType & VideoInfo::CS_YUY2) == VideoInfo::CS_YUY2 )
		{
			const unsigned char *pRefYUY2 = child->GetFrame(n + offset, env)->GetReadPtr();
			int nRefPitchYUY2 = child->GetFrame(n + offset, env)->GetPitch();
			pRef = RefPlanes->GetPtr();
			pRefU = RefPlanes->GetPtrU();
			pRefV = RefPlanes->GetPtrV();
			nRefPitch  = RefPlanes->GetPitch();
			nRefPitchUV  = RefPlanes->GetPitchUV();
			YUY2ToPlanes(pRefYUY2, nRefPitchYUY2, analysisData.nWidth, analysisData.nHeight, 
				pRef, nRefPitch, pRefU, pRefV, nRefPitchUV, isse);
		}
		else
		{
		  pRef = child->GetFrame(n + offset, env)->GetReadPtr(PLANAR_Y);
		  pRefU = child->GetFrame(n + offset, env)->GetReadPtr(PLANAR_U);
		  pRefV = child->GetFrame(n + offset, env)->GetReadPtr(PLANAR_V);
		  nRefPitch = child->GetFrame(n + offset, env)->GetPitch(PLANAR_Y);
		  nRefPitchUV = child->GetFrame(n + offset, env)->GetPitch(PLANAR_U);
		}

      MVFrames *pFrames = mvCore.GetFrames(analysisData.nIdx);

      PROFILE_START(MOTION_PROFILE_INTERPOLATION);
      MVGroupOfFrames *pSrcGOF = pFrames->GetFrame(n);
      pSrcGOF->SetPlane(pSrc, nSrcPitch, YPLANE);
      pSrcGOF->SetPlane(pSrcU, nSrcPitchUV, UPLANE);
      pSrcGOF->SetPlane(pSrcV, nSrcPitchUV, VPLANE);
      pSrcGOF->Reduce();
      pSrcGOF->Pad(YUVPLANES);
      pSrcGOF->Refine(YUVPLANES);

      MVGroupOfFrames *pRefGOF = pFrames->GetFrame(n + offset);
      pRefGOF->SetPlane(pRef, nRefPitch, YPLANE);
      pRefGOF->SetPlane(pRefU, nRefPitchUV, UPLANE);
      pRefGOF->SetPlane(pRefV, nRefPitchUV, VPLANE);
      pRefGOF->Reduce();
      pRefGOF->Pad(YUVPLANES);
      pRefGOF->Refine(YUVPLANES);
      PROFILE_STOP(MOTION_PROFILE_INTERPOLATION);

      vectorFields->SearchMVs(pSrcGOF, pRefGOF, searchType, nSearchParam, nLambda, lsad, pnew, plevel, global, analysisData.nFlags, reinterpret_cast<int*>(pDst));

      PROFILE_CUMULATE();
	}
	else vectorFields->WriteDefaultToArray(reinterpret_cast<int*>(pDst));

	return dst;
}