// Make a motion compensate temporal denoiser

// See legal notice in Copying.txt for more information

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA, or visit
// http://www.gnu.org/copyleft/gpl.html .

#include "MVCompensate.h"
#include "Padding.h"
#include "Interpolation.h"

MVCompensate::MVCompensate(PClip _child, PClip vectors, bool sc, bool inclip, bool inloop, int _nIdx,
                           int nSCD1, int nSCD2, bool _mmx, bool _isse, IScriptEnvironment* env) :
GenericVideoFilter(_child),
mvClip(vectors, nSCD1, nSCD2, env),
MVFilter(vectors, "MVCompensate", env)
{
   isse = _isse;
   mmx = _mmx;
   scBehavior = sc;
   inLoop = inloop;
   inClip = inclip;
   nRefPitch = nWidth + nHPadding * 2;
   nRefWidth = nRefPitch;
   nRefHeight = nHeight + nVPadding * 2;

   nIdx = _nIdx;

   nRefPitchUV = nRefPitch / 2;
   nRefWidthUV = nRefWidth / 2;
   nRefHeightUV = nRefHeight / yRatioUV;

   nOffset = nHPadding + nVPadding * nRefPitch;
   nOffsetUV = nHPadding / 2 + (nVPadding / yRatioUV) * nRefPitchUV;


   if ( !inClip && !inLoop )
      mvClip.SetVectorsNeed(false, false, false, true, true, true);

   if ( inLoop )
   {
      pRefs = new unsigned char *[nPel * nPel];
      pRefsU = new unsigned char *[nPel * nPel];
      pRefsV = new unsigned char *[nPel * nPel];
      for ( int i = 0; i < nPel * nPel; i++ )
      {
         pRefs[i] = new unsigned char[nRefPitch * nRefHeight];
         pRefsU[i] = new unsigned char[nRefPitchUV * nRefHeightUV];
         pRefsV[i] = new unsigned char[nRefPitchUV * nRefHeightUV];
      }
   }

   if ( isse )
   {
      switch (nBlkSize)
      {
      case 16:
         BLITLUMA = Copy16_mmx;
		 if (yRatioUV==2)
	         BLITCHROMA = Copy8_mmx;
		 else//yRatioUV==1
	         BLITCHROMA = Copy8x16_mmx;
         break;
      case 4:
		 BLITLUMA = Copy4_mmx;
		 if (yRatioUV==2) {
			 BLITCHROMA = Copy2_mmx;
		 }
		 else { //yRatioUV==1
			 BLITCHROMA = Copy_mmx<2, 4>;
		 }
         break;
      case 8:
      default:
         BLITLUMA = Copy8_mmx;
		 if (yRatioUV==2) {
	         BLITCHROMA = Copy4_mmx;
		 }
		 else {//yRatioUV==1
	         BLITCHROMA = Copy4x8_mmx;
		 }
      }
   }
   else
   {
      switch (nBlkSize)
      {
      case 16:
         BLITLUMA = Copy_C<16>; // "mmx" version could be used, but it's more like a debugging version
		 if (yRatioUV==2)
	         BLITCHROMA = Copy_C<8>; // idem
		 else//yRatioUV==1
	         BLITCHROMA = Copy_C<8,16>; // idem
         break;
      case 4:
         BLITLUMA = Copy_C<4>; // "mmx" version could be used, but it's more like a debugging version
		 if (yRatioUV==2)
	         BLITCHROMA = Copy_C<2>; // idem
		 else//yRatioUV==1
	         BLITCHROMA = Copy_C<2,4>; // idem
         break;
      case 8:
      default:
         BLITLUMA = Copy_C<8>;
		 if (yRatioUV==2) {
			 BLITCHROMA = Copy_C<4>; // idem
		 }
		 else {//yRatioUV==1
			 BLITCHROMA = Copy_C<4,8>; // idem
		 }
      }
   }

   if ( inClip )
      mvCore->AddFrames(nIdx, MV_BUFFER_FRAMES, mvClip.GetLevelCount(), nWidth, nHeight,
                        nPel, nHPadding, nVPadding, YUVPLANES, isse, yRatioUV);
   if ( (pixelType & VideoInfo::CS_YUY2) == VideoInfo::CS_YUY2 )
   {
		RefPlanes =  new YUY2Planes(nWidth, nHeight);
		DstPlanes =  new YUY2Planes(nWidth, nHeight);
   }
}

MVCompensate::~MVCompensate()
{
   if ( (pixelType & VideoInfo::CS_YUY2) == VideoInfo::CS_YUY2 )
   {
	delete RefPlanes;
	delete DstPlanes;
   }
   if ( inLoop )
   {
      for ( int i = 0; i < nPel * nPel; i++ )
      {
         delete[] pRefs[i];
         delete[] pRefsU[i];
         delete[] pRefsV[i];
      }
      delete[] pRefs;
      delete[] pRefsU;
      delete[] pRefsV;
   }
}

void MVCompensate::MoveBlock(unsigned char *pDst, int nDstPitch, int x, int y, int mvx, int mvy)
{
   pDst += x + y * nDstPitch;
   unsigned char *pSrc;
   if ( nPel == 1 )
      pSrc = pRefs[0] + x + mvx + nHPadding + (y + mvy + nVPadding) * nRefPitch;
   else
   {
	int idx = (mvx & 1) + (mvy & 1) * 2;
      pSrc = pRefs[idx] + (x * 2 + mvx + nHPadding * 2) / 2 + ((y * 2 + mvy + nVPadding * 2) / 2) * nRefPitch;
   }
   for ( int i = 0; i < nBlkSize; i++ )
   {
      memcpy(pDst, pSrc, nBlkSize);
      pDst += nDstPitch;
      pSrc += nRefPitch;
   }
}

void MVCompensate::MoveBlockU(unsigned char *pDst, int nDstPitch, int x, int y, int mvx, int mvy)
{
   pDst += (x / 2) + (y / yRatioUV) * nDstPitch;
   unsigned char *pSrc;
   if ( nPel == 1 )
      pSrc = pRefsU[0] + (x + mvx + nHPadding) / 2 + ((y + mvy + nVPadding) / yRatioUV) * nRefPitchUV;
   else
   {
	int idx = (mvx & 1) + (mvy & 1) * 2;
      pSrc = pRefsU[idx] + (x * 2 + mvx + nHPadding * 2) / 4 + ((y * 2 + mvy + nVPadding * 2) / (2*yRatioUV)) * nRefPitchUV;
   }
   for ( int i = 0; i < nBlkSize / yRatioUV; i++ )
   {
      memcpy(pDst, pSrc, nBlkSize / 2);
      pDst += nDstPitch;
      pSrc += nRefPitchUV;
   }
}

void MVCompensate::MoveBlockV(unsigned char *pDst, int nDstPitch, int x, int y, int mvx, int mvy)
{
   pDst += (x / 2) + (y / yRatioUV) * nDstPitch;
   unsigned char *pSrc;
   if ( nPel == 1 )
      pSrc = pRefsV[0] + (x + mvx + nHPadding) / 2 + ((y + mvy + nVPadding) / yRatioUV) * nRefPitchUV;
   else
   {
	int idx = (mvx & 1) + (mvy & 1) * 2;
      pSrc = pRefsV[idx] + (x * 2 + mvx + nHPadding * 2) / 4 + ((y * 2 + mvy + nVPadding * 2) / (2*yRatioUV)) * nRefPitchUV;
   }
   for ( int i = 0; i < nBlkSize / yRatioUV; i++ )
   {
      memcpy(pDst, pSrc, nBlkSize / 2);
      pDst += nDstPitch;
      pSrc += nRefPitchUV;
   }
}

PVideoFrame __stdcall MVCompensate::GetFrame(int n, IScriptEnvironment* env)
{
	int nWidth_B = nBlkX*(nBlkSize - nOverlap) + nOverlap;
	int nHeight_B = nBlkY*(nBlkSize - nOverlap) + nOverlap;

	PVideoFrame	src	= child->GetFrame(n, env);
   PVideoFrame dst;
   BYTE *pDst[3], *pDstCur[3];
	const BYTE *pRef[3];
    int nDstPitches[3], nRefPitches[3];
	unsigned char *pDstYUY2;
	int nDstPitchYUY2;

   mvClip.Update(n, env);

   int off = ( mvClip.IsBackward() ) ? 1 : -1;
   off *= mvClip.GetDeltaFrame();

   if ( mvClip.IsUsable() )
   {
      dst = env->NewVideoFrame(vi);

  		if ( (pixelType & VideoInfo::CS_YUY2) == VideoInfo::CS_YUY2 )
		{
			pDstYUY2 = dst->GetWritePtr();
			nDstPitchYUY2 = dst->GetPitch();
			pDst[0] = DstPlanes->GetPtr();
			pDst[1] = DstPlanes->GetPtrU();
			pDst[2] = DstPlanes->GetPtrV();
			nDstPitches[0]  = DstPlanes->GetPitch();
			nDstPitches[1]  = DstPlanes->GetPitchUV();
			nDstPitches[2]  = DstPlanes->GetPitchUV();
			YUY2ToPlanes(pDstYUY2, nDstPitchYUY2, nWidth, nHeight, 
				pDst[0], nDstPitches[0], pDst[1], pDst[2], nDstPitches[1], isse);
		}
		else
		{
			 pDst[0] = YWPLAN(dst);
			 pDst[1] = UWPLAN(dst);
			 pDst[2] = VWPLAN(dst);
			 nDstPitches[0] = YPITCH(dst);
			 nDstPitches[1] = UPITCH(dst);
			 nDstPitches[2] = VPITCH(dst);
		}


      if ( inLoop )
      {
         // Pad the reference frame
         Padding::PadReferenceFrame(pRefs[0], nRefPitch, nHPadding, nVPadding, nWidth, nHeight);
         Padding::PadReferenceFrame(pRefsU[0], nRefPitchUV, nHPadding / 2, nVPadding / yRatioUV, nWidth / 2, nHeight / yRatioUV);
         Padding::PadReferenceFrame(pRefsV[0], nRefPitchUV, nHPadding / 2, nVPadding / yRatioUV, nWidth / 2, nHeight / yRatioUV);

         // Resample the reference frame
         if ( nPel == 2 )
         {
            VerticalBilin(pRefs[2], pRefs[0], nRefPitch, nRefPitch, nRefWidth, nRefHeight);
            VerticalBilin(pRefsU[2], pRefsU[0], nRefPitchUV, nRefPitchUV, nRefWidthUV, nRefHeightUV);
            VerticalBilin(pRefsV[2], pRefsV[0], nRefPitchUV, nRefPitchUV, nRefWidthUV, nRefHeightUV);
            HorizontalBilin(pRefs[1], pRefs[0], nRefPitch, nRefPitch, nRefWidth, nRefHeight);
            HorizontalBilin(pRefsU[1], pRefsU[0], nRefPitchUV, nRefPitchUV, nRefWidthUV, nRefHeightUV);
            HorizontalBilin(pRefsV[1], pRefsV[0], nRefPitchUV, nRefPitchUV, nRefWidthUV, nRefHeightUV);
            DiagonalBilin(pRefs[3], pRefs[0], nRefPitch, nRefPitch, nRefWidth, nRefHeight);
            DiagonalBilin(pRefsU[3], pRefsU[0], nRefPitchUV, nRefPitchUV, nRefWidthUV, nRefHeightUV);
            DiagonalBilin(pRefsV[3], pRefsV[0], nRefPitchUV, nRefPitchUV, nRefWidthUV, nRefHeightUV);
         }

         // Makes compensation
         for ( int i = 0; i < mvClip.GetBlkCount(); i++ )
         {
            MoveBlock(pDst[0], nDstPitches[0], mvClip.GetBlock(0, i).GetX(),
               mvClip.GetBlock(0, i).GetY(), mvClip.GetBlock(0, i).GetMV().x, mvClip.GetBlock(0, i).GetMV().y);
            MoveBlockU(pDst[1], nDstPitches[1], mvClip.GetBlock(0, i).GetX(),
               mvClip.GetBlock(0, i).GetY(), mvClip.GetBlock(0, i).GetMV().x, mvClip.GetBlock(0, i).GetMV().y);
            MoveBlockV(pDst[2], nDstPitches[2], mvClip.GetBlock(0, i).GetX(),
               mvClip.GetBlock(0, i).GetY(), mvClip.GetBlock(0, i).GetMV().x, mvClip.GetBlock(0, i).GetMV().y);
         }

         // Moves the new frame into pRefs[0], if we're in inloop mode
         if ( inLoop )
         {
            env->BitBlt(pRefs[0] + nOffset, nRefWidth, pDst[0], nDstPitches[0],
               nWidth, nHeight);
            env->BitBlt(pRefsU[0] + nOffsetUV, nRefWidthUV, pDst[1], nDstPitches[1],
               nWidth / 2, nHeight / yRatioUV);
            env->BitBlt(pRefsV[0] + nOffsetUV, nRefWidthUV, pDst[2], nDstPitches[2],
               nWidth / 2, nHeight / yRatioUV);
         }
      }
      else if ( inClip )
      {
         PVideoFrame ref = child->GetFrame(n + off, env);
         MVFrames *pFrames = mvCore->GetFrames(nIdx);
         MVGroupOfFrames *pRefGOF = pFrames->GetFrame(n + off);

  		if ( (pixelType & VideoInfo::CS_YUY2) == VideoInfo::CS_YUY2 )
		{
			const unsigned char *pRefYUY2 = ref->GetReadPtr();
			int nRefPitchYUY2 = ref->GetPitch();
			pRef[0] = RefPlanes->GetPtr();
			pRef[1] = RefPlanes->GetPtrU();
			pRef[2] = RefPlanes->GetPtrV();
			nRefPitches[0]  = RefPlanes->GetPitch();
			nRefPitches[1]  = RefPlanes->GetPitchUV();
			nRefPitches[2]  = RefPlanes->GetPitchUV();
			YUY2ToPlanes(pRefYUY2, nRefPitchYUY2, nWidth, nHeight, 
				pRef[0], nRefPitches[0], pRef[1], pRef[2], nRefPitches[1], isse);
		}
		else
		{
			 pRef[0] = YRPLAN(ref);
			 pRef[1] = URPLAN(ref);
			 pRef[2] = VRPLAN(ref);
			 nRefPitches[0] = YPITCH(ref);
			 nRefPitches[1] = UPITCH(ref);
			 nRefPitches[2] = VPITCH(ref);
		}
         PROFILE_START(MOTION_PROFILE_INTERPOLATION);

         pRefGOF->SetPlane(pRef[0], nRefPitches[0], YPLANE);
         pRefGOF->SetPlane(pRef[1], nRefPitches[1], UPLANE);
         pRefGOF->SetPlane(pRef[2], nRefPitches[2], VPLANE);
         pRefGOF->Pad(YUVPLANES);
         pRefGOF->Refine(YUVPLANES);

         PROFILE_STOP(MOTION_PROFILE_INTERPOLATION);

         MVPlane *pPlanes[3];

         pPlanes[0] = pRefGOF->GetFrame(0)->GetPlane(YPLANE);
         pPlanes[1] = pRefGOF->GetFrame(0)->GetPlane(UPLANE);
         pPlanes[2] = pRefGOF->GetFrame(0)->GetPlane(VPLANE);

         PROFILE_START(MOTION_PROFILE_COMPENSATION);
		 pDstCur[0] = pDst[0];
		 pDstCur[1] = pDst[1];
		 pDstCur[2] = pDst[2];
//         for ( int i = 0; i < mvClip.GetBlkCount(); i++ )
		for (int by=0; by<nBlkY; by++)
		{
			if (by*(nBlkSize - nOverlap) + nBlkSize > nHeight) break;
			int xx = 0;
			for (int bx=0; bx<nBlkX; bx++)
			{
				if ( xx + nBlkSize <= nWidth)
				{
					int i = by*nBlkX + bx;
					const FakeBlockData &block = mvClip.GetBlock(0, i);

					// luma
					BLITLUMA(pDstCur[0] + xx, nDstPitches[0],
					   pPlanes[0]->GetPointer(block.GetX() * nPel + block.GetMV().x, block.GetY() * nPel + block.GetMV().y), 
					   pPlanes[0]->GetPitch());

					// chroma u
					BLITCHROMA(pDstCur[1] + (xx>>1), nDstPitches[1],
					   pPlanes[1]->GetPointer((block.GetX() * nPel + block.GetMV().x) >> 1, (block.GetY() * nPel + block.GetMV().y) /yRatioUV),
					   pPlanes[1]->GetPitch());

					// chroma v
					BLITCHROMA(pDstCur[2] + (xx>>1), nDstPitches[2],
					   pPlanes[2]->GetPointer((block.GetX() * nPel + block.GetMV().x) >> 1, (block.GetY() * nPel + block.GetMV().y) /yRatioUV),
					   pPlanes[2]->GetPitch());

					// update pDsts
//					pDst[0] += (nBlkSize - nOverlap);
//					pDst[1] += (nBlkSize - nOverlap) >> 1;
//					pDst[2] += (nBlkSize - nOverlap) >> 1;
					xx += (nBlkSize - nOverlap);
				}
			}
//            if ( !((i + 1) % nBlkX)  )
//            {
               pDstCur[0] += (nBlkSize - nOverlap) * (nDstPitches[0]); // Fizick
               pDstCur[1] += ( nBlkSize  - nOverlap) * (nDstPitches[1]) /yRatioUV;
               pDstCur[2] += ( nBlkSize  - nOverlap) * (nDstPitches[2]) /yRatioUV;
//            }
			
		}
		 __asm emms; // (we may use double-float somewhere) Fizick
         PROFILE_STOP(MOTION_PROFILE_COMPENSATION);
      }
      else
      {
         PROFILE_START(MOTION_PROFILE_COMPENSATION);

         // we're in the default mode : compensation has been made while searching the mvs
		env->BitBlt(pDst[0], nDstPitches[0], 
               mvClip.GetCompensatedPlane(), mvClip.GetPitch(), nWidth_B, nHeight_B);
         env->BitBlt(pDst[1], nDstPitches[1], 
               mvClip.GetCompensatedPlaneU(), mvClip.GetPitchUV(), nWidth_B / 2, nHeight_B/yRatioUV);
         env->BitBlt(pDst[2], nDstPitches[2], 
               mvClip.GetCompensatedPlaneV(), mvClip.GetPitchUV(), nWidth_B / 2, nHeight_B/yRatioUV);

         PROFILE_STOP(MOTION_PROFILE_COMPENSATION);
      }

	if ( (pixelType & VideoInfo::CS_YUY2) == VideoInfo::CS_YUY2 )
	{
		YUY2FromPlanes(pDstYUY2, nDstPitchYUY2, nWidth_B, nHeight_B,
					  pDst[0], nDstPitches[0], pDst[1], pDst[2], nDstPitches[1], isse);
	}
   }
   else {
      if ( !scBehavior && ( n + off < vi.num_frames ) && ( n + off >= 0 ))
         dst = child->GetFrame(n + off, env);
      else
         dst = src;


      // if we're in in-loop mode, we copy the frame into pRefs[0]
      if ( inLoop )
      {
  		if ( (pixelType & VideoInfo::CS_YUY2) == VideoInfo::CS_YUY2 )
		{
			pDstYUY2 = (BYTE *)dst->GetReadPtr();
			nDstPitchYUY2 = dst->GetPitch();
			pDst[0] = DstPlanes->GetPtr();
			pDst[1] = DstPlanes->GetPtrU();
			pDst[2] = DstPlanes->GetPtrV();
			nDstPitches[0]  = DstPlanes->GetPitch();
			nDstPitches[1]  = DstPlanes->GetPitchUV();
			nDstPitches[2]  = DstPlanes->GetPitchUV();
			YUY2ToPlanes(pDstYUY2, nDstPitchYUY2, nWidth, nHeight, 
				pDst[0], nDstPitches[0], pDst[1], pDst[2], nDstPitches[1], isse);
		}
		else
		{
		 pDst[0] = (BYTE *)YRPLAN(dst);
         pDst[1] = (BYTE *)URPLAN(dst);
         pDst[2] = (BYTE *)VRPLAN(dst);
         nDstPitches[0] = YPITCH(dst);
         nDstPitches[1] = UPITCH(dst);
         nDstPitches[2] = VPITCH(dst);
		}
        env->BitBlt(pRefs[0] + nOffset, nRefWidth, pDst[0], nDstPitches[0],
               nWidth, nHeight);
         env->BitBlt(pRefsU[0] + nOffsetUV, nRefWidthUV, pDst[1], nDstPitches[1],
               nWidth / 2, nHeight / yRatioUV);
         env->BitBlt(pRefsV[0] + nOffsetUV, nRefWidthUV, pDst[2], nDstPitches[2],
               nWidth / 2, nHeight / yRatioUV);
      }
   }
   return dst;
}