
// Make a motion compensate temporal denoiser

// See legal notice in Copying.txt for more information

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA, or visit
// http://www.gnu.org/copyleft/gpl.html .

#include "MVDenoise2.h"
#include "CopyCode.h"
#include "Padding.h"

MVDenoise2::MVDenoise2(PClip _child, AVSValue mvArray, int _nMode, int thT, int sadT, int thMV, int nSCD1, int nSCD2, bool mmx, bool _isse, IScriptEnvironment* env) :
GenericVideoFilter(_child),
mvClipArray(mvArray, nSCD1, nSCD2, env),
MVFilter(mvArray[0].AsClip(), "MVDenoise", env)
{
   nMode = _nMode;
   isse = _isse;

   DebugPrintf("Constructing MVDenoise...");

   for ( int i = 0; i < mvClipArray.size(); i++  )
   {
      CheckSimilarity(mvClipArray[i], "vector", env);
      mvClipArray[i].SetVectorsNeed(true, true, false, nMode & YPLANE, nMode & UPLANE, nMode & VPLANE);
   }

 	/*
		We give the thresholds in a comprehensive way, so adjustments have to be made
	*/

   if ( !(int)mvClipArray.size() )
      env->ThrowError("MVDenoise : you must provide at least one vectors stream");

	TemporalThreshold = thT;
	if ( TemporalThreshold < 0 ) TemporalThreshold = 0;

	SADThreshold = sadT;

	MVThreshold = thMV * thMV * nPel * nPel;

	pitch =  vi.width + 2 * nHPadding;

	offset = nVPadding * pitch + nHPadding;

   size = pitch * ( vi.height + 2 * nVPadding );

   if ( (pixelType & VideoInfo::CS_YUY2) == VideoInfo::CS_YUY2 )
   {
		DstPlanes =  new YUY2Planes(nWidth, nHeight);
   }
}

MVDenoise2::~MVDenoise2()
{
   if ( (pixelType & VideoInfo::CS_YUY2) == VideoInfo::CS_YUY2 )
   {
	delete DstPlanes;
   }
}

#define SATURATE(x,a,b) (((x) < (a)) ? (a) : (((x) > (b)) ? (b) : (x)))

void MVDenoise2::DenoiseBlockLuma(unsigned char *pDst, int nDstPitch)
{
   bool *dos = new bool[(int)mvClipArray.size()];
   const unsigned char **pRefs = new const unsigned char *[(int)mvClipArray.size()];
   int *pitches = new int[(int)mvClipArray.size()];
   int *pLumas = new int[(int)mvClipArray.size()];

   int nLumaRatio = nBlkSize * nBlkSize;

   for ( int j = 0; j < (int)mvClipArray.size(); j++ )
      dos[j] = mvClipArray[j].IsUsable();
 
	for ( int i = 0; i < nBlkCount; i++ )
	{
      int x = mvClipArray[0].GetBlock(0, i).GetX();
		int y = mvClipArray[0].GetBlock(0, i).GetY();
      unsigned char *d = pDst + x + y * nDstPitch;
      for ( int j = 0; j < (int)mvClipArray.size(); j++ )
      {
         if (dos[j] && (mvClipArray[j].GetBlock(0, i).GetSAD() < SADThreshold) && (mvClipArray[j].GetBlock(0, i).GetMVLength() < MVThreshold))
         {
            pitches[j] = mvClipArray[j].GetPitch();
            pRefs[j] = mvClipArray[j].GetCompensatedPlane() + x + y * pitches[j];
            pLumas[j] = (mvClipArray[j].GetBlock(0, i).GetLuma() - mvClipArray[j].GetBlock(0, i).GetRefLuma()) / nLumaRatio;
         }
         else
         {
            pitches[j] = nDstPitch;
            pRefs[j] = d;
            pLumas[j] = 0;
         }
      }
      for ( int k = 0; k < nBlkSize; k++ )
		{
			for ( int l = 0; l < nBlkSize; l++ )
			{
				int count = 1;
            int newpix = d[l];
            for ( int j = 0; j < (int)mvClipArray.size(); j++ )
            {
               int v = SATURATE(pRefs[j][l] + pLumas[j], 0, 255);
    		      if (MABS(d[l] - v) < TemporalThreshold)
	    	      {
		    	      newpix += v;
				      count += 1;
			      }
            }
	   		d[l] = (newpix + (count >> 1)) / count;
	   	}
	   	d += nDstPitch;
         for ( int j = 0; j < (int)mvClipArray.size(); j++ )
            pRefs[j] += pitches[j];
		}
	}
   delete[] dos;
   delete[] pitches;
   delete[] pRefs;
   delete[] pLumas;
}

void MVDenoise2::DenoiseBlockChromaU(unsigned char *pDst, int nDstPitch)
{
   bool *dos = new bool[(int)mvClipArray.size()];
   const unsigned char **pRefs = new const unsigned char *[(int)mvClipArray.size()];
   int *pitches = new int[(int)mvClipArray.size()];

   for ( int j = 0; j < (int)mvClipArray.size(); j++ )
      dos[j] = mvClipArray[j].IsUsable();

	for ( int i = 0; i < nBlkCount; i++ )
	{
      int x = mvClipArray[0].GetBlock(0, i).GetX() >> 1;
		int y = mvClipArray[0].GetBlock(0, i).GetY() /yRatioUV;
      unsigned char *d = pDst + x + y * nDstPitch;
      for ( int j = 0; j < (int)mvClipArray.size(); j++ )
      {
         if (dos[j] && (mvClipArray[j].GetBlock(0, i).GetSAD() < SADThreshold) && (mvClipArray[j].GetBlock(0, i).GetMVLength() < MVThreshold))
         {
            pitches[j] = mvClipArray[j].GetPitchUV();
            pRefs[j] = mvClipArray[j].GetCompensatedPlaneU() + x + y * pitches[j];
         }
         else
         {
            pitches[j] = nDstPitch;
            pRefs[j] = d;
         }
      }
      for ( int k = 0; k < nBlkSize /yRatioUV; k++ )
		{
			for ( int l = 0; l < nBlkSize >> 1; l++ )
			{
				int count = 1;
            int newpix = d[l];
            for ( int j = 0; j < (int)mvClipArray.size(); j++ )
            {
               int v = SATURATE(pRefs[j][l], 0, 255);
    		      if (MABS(d[l] - v) < TemporalThreshold)
	    	      {
		    	      newpix += v;
				      count += 1;
			      }
            }
	   		d[l] = (newpix + (count >> 1)) / count;
	   	}
	   	d += nDstPitch;
         for ( int j = 0; j < (int)mvClipArray.size(); j++ )
            pRefs[j] += pitches[j];
		}
	}
   delete[] dos;
   delete[] pitches;
   delete[] pRefs;
}

void MVDenoise2::DenoiseBlockChromaV(unsigned char *pDst, int nDstPitch)
{
   bool *dos = new bool[(int)mvClipArray.size()];
   const unsigned char **pRefs = new const unsigned char *[(int)mvClipArray.size()];
   int *pitches = new int[(int)mvClipArray.size()];

   for ( int j = 0; j < (int)mvClipArray.size(); j++ )
      dos[j] = mvClipArray[j].IsUsable();
 
	for ( int i = 0; i < nBlkCount; i++ )
	{
      int x = mvClipArray[0].GetBlock(0, i).GetX() >> 1;
		int y = mvClipArray[0].GetBlock(0, i).GetY() /yRatioUV;
      unsigned char *d = pDst + x + y * nDstPitch;
      for ( int j = 0; j < (int)mvClipArray.size(); j++ )
      {
         if (dos[j] && (mvClipArray[j].GetBlock(0, i).GetSAD() < SADThreshold) && (mvClipArray[j].GetBlock(0, i).GetMVLength() < MVThreshold))
         {
            pitches[j] = mvClipArray[j].GetPitchUV();
            pRefs[j] = mvClipArray[j].GetCompensatedPlaneV() + x + y * pitches[j];
         }
         else
         {
            pitches[j] = nDstPitch;
            pRefs[j] = d;
         }
      }
      for ( int k = 0; k < nBlkSize /yRatioUV; k++ )
		{
			for ( int l = 0; l < nBlkSize >> 1; l++ )
			{
				int count = 1;
            int newpix = d[l];
            for ( int j = 0; j < (int)mvClipArray.size(); j++ )
            {
               int v = SATURATE(pRefs[j][l], 0, 255);
    		      if (MABS(d[l] - v) < TemporalThreshold)
	    	      {
		    	      newpix += v;
				      count += 1;
			      }
            }
	   		d[l] = (newpix + (count >> 1)) / count;
	   	}
	   	d += nDstPitch;
         for ( int j = 0; j < (int)mvClipArray.size(); j++ )
            pRefs[j] += pitches[j];
		}
	}
   delete[] dos;
   delete[] pitches;
   delete[] pRefs;
}

PVideoFrame __stdcall MVDenoise2::GetFrame(int n, IScriptEnvironment* env)
{
   BYTE *pDst[3];
    int nDstPitches[3];
	const unsigned char *pSrcYUY2;
	unsigned char *pDstYUY2;
	int nDstPitchYUY2, nSrcPitchYUY2;

	PVideoFrame	src = child->GetFrame(n, env);
	PVideoFrame	dst = env->NewVideoFrame(vi);

  		if ( (pixelType & VideoInfo::CS_YUY2) == VideoInfo::CS_YUY2 )
		{
			pSrcYUY2 = src->GetReadPtr();
			nSrcPitchYUY2 = src->GetPitch();
			pDstYUY2 = dst->GetWritePtr();
			nDstPitchYUY2 = dst->GetPitch();
			pDst[0] = DstPlanes->GetPtr();
			pDst[1] = DstPlanes->GetPtrU();
			pDst[2] = DstPlanes->GetPtrV();
			nDstPitches[0]  = DstPlanes->GetPitch();
			nDstPitches[1]  = DstPlanes->GetPitchUV();
			nDstPitches[2]  = DstPlanes->GetPitchUV();
			YUY2ToPlanes(pSrcYUY2, nSrcPitchYUY2, nWidth, nHeight, 
				pDst[0], nDstPitches[0], pDst[1], pDst[2], nDstPitches[1], isse);
		}
		else
		{
			 pDst[0] = YWPLAN(dst);
			 pDst[1] = UWPLAN(dst);
			 pDst[2] = VWPLAN(dst);
			 nDstPitches[0] = YPITCH(dst);
			 nDstPitches[1] = UPITCH(dst);
			 nDstPitches[2] = VPITCH(dst);
			env->BitBlt(pDst[0], nDstPitches[0], YRPLAN(src), YPITCH(src), nWidth, nHeight);
			env->BitBlt(pDst[1], nDstPitches[1], URPLAN(src), UPITCH(src), nWidth >> 1, nHeight / yRatioUV);
			env->BitBlt(pDst[2], nDstPitches[2], VRPLAN(src), VPITCH(src), nWidth >> 1, nHeight / yRatioUV);
		}


   for ( int j = 0; j < (int)mvClipArray.size(); j++ )
      mvClipArray[j].Update(n, env);

   if ( nMode & YPLANE )
      DenoiseBlockLuma(pDst[0], nDstPitches[0]);
   if ( nMode & UPLANE )
      DenoiseBlockChromaU(pDst[1], nDstPitches[1]);
   if ( nMode & VPLANE )
      DenoiseBlockChromaV(pDst[2], nDstPitches[2]);

	if ( (pixelType & VideoInfo::CS_YUY2) == VideoInfo::CS_YUY2 )
	{
		YUY2FromPlanes(pDstYUY2, nDstPitchYUY2, nWidth, nHeight,
					  pDst[0], nDstPitches[0], pDst[1], pDst[2], nDstPitches[1], isse);
	}
	return dst;
}