#include "MVInterface.h"
#include "Padding.h"
#include "CopyCode.h"
#include "Interpolation.h"

/******************************************************************************
*                                                                             *
*  MVPlane : manages a single plane, allowing padding and refinin             *
*                                                                             *
******************************************************************************/

MVPlane::MVPlane(int _nWidth, int _nHeight, int _nPel, int _nHPad, int _nVPad, bool _isse)
{
   nWidth = _nWidth;
   nHeight = _nHeight;
   nPel = _nPel;
   nHPadding = _nHPad;
   nVPadding = _nVPad;
   isse = _isse;

   nExtendedWidth = nWidth + 2 * nHPadding;
   nExtendedHeight = nHeight + 2 * nVPadding;

   nPitch = (nExtendedWidth + 15) & (~15);
   nOffsetPadding = nPitch * nVPadding + nHPadding;

   pPlane = new Uint8 *[nPel * nPel];
   for ( int i = 0; i < nPel * nPel; i++ )
      pPlane[i] = new Uint8[nPitch * nExtendedHeight];
}

MVPlane::~MVPlane()
{
   for ( int i = 0; i < nPel * nPel; i++ )
      delete[] pPlane[i];
   delete[] pPlane;
}

void MVPlane::ChangePlane(const Uint8 *pNewPlane, int nNewPitch)
{
   if ( !isFilled )
      BitBlt(pPlane[0] + nOffsetPadding, nPitch, pNewPlane, nNewPitch, nWidth, nHeight, isse);
   isFilled = true;
}

void MVPlane::Pad()
{
   if ( !isPadded )
      Padding::PadReferenceFrame(pPlane[0], nPitch, nHPadding, nVPadding, nWidth, nHeight);
   isPadded = true;
}

void MVPlane::Refine()
{
   if (( nPel == 2 ) && ( !isRefined ))
   {
      if (isse)
      {
         HorizontalBilin_iSSE(pPlane[1], pPlane[0], nPitch, nPitch, nExtendedWidth, nExtendedHeight);
         VerticalBilin_iSSE(pPlane[2], pPlane[0], nPitch, nPitch, nExtendedWidth, nExtendedHeight);
         DiagonalBilin_iSSE(pPlane[3], pPlane[0], nPitch, nPitch, nExtendedWidth, nExtendedHeight);
      }
      else
      {
         HorizontalBilin(pPlane[1], pPlane[0], nPitch, nPitch, nExtendedWidth, nExtendedHeight);
         VerticalBilin(pPlane[2], pPlane[0], nPitch, nPitch, nExtendedWidth, nExtendedHeight);
         DiagonalBilin(pPlane[3], pPlane[0], nPitch, nPitch, nExtendedWidth, nExtendedHeight);
      }
   }
   isRefined = true;
}

void MVPlane::ReduceTo(MVPlane *pReducedPlane)
{
   if ( !pReducedPlane->isFilled )
   {
      if (isse)
      {
         RB2F_iSSE(pReducedPlane->pPlane[0] + pReducedPlane->nOffsetPadding, pPlane[0] + nOffsetPadding, 
            pReducedPlane->nPitch, nPitch, pReducedPlane->nWidth, pReducedPlane->nHeight);
      }
      else
      {
         RB2F_C(pReducedPlane->pPlane[0] + pReducedPlane->nOffsetPadding, pPlane[0] + nOffsetPadding, 
            pReducedPlane->nPitch, nPitch, pReducedPlane->nWidth, pReducedPlane->nHeight);
      }
   }
   pReducedPlane->isFilled = true;
}

void MVPlane::WritePlane(FILE *pFile)
{
   for ( int i = 0; i < nHeight; i++ )
      fwrite(pPlane[0] + i * nPitch + nOffsetPadding, 1, nWidth, pFile);
}

/******************************************************************************
*                                                                             *
*  MVFrame : a MVFrame is a threesome of MVPlane, some undefined, some        *
*  defined, according to the nMode value                                      *
*                                                                             *
******************************************************************************/

MVFrame::MVFrame(int nWidth, int nHeight, int nPel, int nHPad, int nVPad, int _nMode, bool _isse, int _yRatioUV)
{
   nMode = _nMode;
   isse = _isse;
   yRatioUV = _yRatioUV;

   if ( nMode & YPLANE )
      pYPlane = new MVPlane(nWidth, nHeight, nPel, nHPad, nVPad, isse);
   else
      pYPlane = 0;

   if ( nMode & UPLANE )
      pUPlane = new MVPlane(nWidth / 2, nHeight / yRatioUV, nPel, nHPad / 2, nVPad / yRatioUV, isse);
   else
      pUPlane = 0;

   if ( nMode & VPLANE )
      pVPlane = new MVPlane(nWidth / 2, nHeight / yRatioUV, nPel, nHPad / 2, nVPad / yRatioUV, isse);
   else
      pVPlane = 0;
}

MVFrame::~MVFrame()
{
   if ( nMode & YPLANE )
      delete pYPlane;

   if ( nMode & UPLANE )
      delete pUPlane;

   if ( nMode & VPLANE )
      delete pVPlane;
}

void MVFrame::ChangePlane(const unsigned char *pNewPlane, int nNewPitch, MVPlaneSet _nMode)
{
   if ( _nMode & nMode & YPLANE )
      pYPlane->ChangePlane(pNewPlane, nNewPitch);

   if ( _nMode & nMode & UPLANE )
      pUPlane->ChangePlane(pNewPlane, nNewPitch);

   if ( _nMode & nMode & VPLANE )
      pVPlane->ChangePlane(pNewPlane, nNewPitch);
}

void MVFrame::Refine(MVPlaneSet _nMode)
{
   if (nMode & YPLANE & _nMode)
      pYPlane->Refine();

   if (nMode & UPLANE & _nMode)
      pUPlane->Refine();

   if (nMode & VPLANE & _nMode)
      pVPlane->Refine();
}

void MVFrame::Pad(MVPlaneSet _nMode)
{
   if (nMode & YPLANE & _nMode)
      pYPlane->Pad();

   if (nMode & UPLANE & _nMode)
      pUPlane->Pad();

   if (nMode & VPLANE & _nMode)
      pVPlane->Pad();
}

void MVFrame::ResetState()
{
   if ( nMode & YPLANE )
      pYPlane->ResetState();

   if ( nMode & UPLANE )
      pUPlane->ResetState();

   if ( nMode & VPLANE )
      pVPlane->ResetState();
}

void MVFrame::WriteFrame(FILE *pFile)
{
   if ( nMode & YPLANE )
      pYPlane->WritePlane(pFile);

   if ( nMode & UPLANE )
      pUPlane->WritePlane(pFile);

   if ( nMode & VPLANE )
      pVPlane->WritePlane(pFile);
}

void MVFrame::ReduceTo(MVFrame *pFrame)
{
   if ( nMode & YPLANE )
      pYPlane->ReduceTo(pFrame->GetPlane(YPLANE));

   if ( nMode & UPLANE )
      pUPlane->ReduceTo(pFrame->GetPlane(UPLANE));

   if ( nMode & VPLANE )
      pVPlane->ReduceTo(pFrame->GetPlane(VPLANE));
}

/******************************************************************************
*                                                                             *
*  MVGroupOfFrames : manage a hierachal frame structure                       *
*                                                                             *
******************************************************************************/

MVGroupOfFrames::MVGroupOfFrames(int _nLevelCount, int nWidth, int nHeight, int nPel, int nHPad, int nVPad, int nMode, bool isse, int yRatioUV)
{
   nLevelCount = _nLevelCount;
   nRefCount = 0;
   nFrameIdx = -1;
   pFrames = new MVFrame *[nLevelCount];

   pFrames[0] = new MVFrame(nWidth, nHeight, nPel, nHPad, nVPad, nMode, isse, yRatioUV);
   for ( int i = 1; i < nLevelCount; i++ )
   {
      nWidth /= 2;
      nHeight /= 2;
      pFrames[i] = new MVFrame(nWidth, nHeight, 1, nHPad, nVPad, nMode, isse, yRatioUV);
   }
}

MVGroupOfFrames::~MVGroupOfFrames()
{
   for ( int i = 0; i < nLevelCount; i++ )
      delete pFrames[i];

   delete[] pFrames;
}

MVFrame *MVGroupOfFrames::GetFrame(int nLevel)
{
   if (( nLevel < 0 ) || ( nLevel >= nLevelCount )) return 0;
   return pFrames[nLevel];
}

void MVGroupOfFrames::SetPlane(const Uint8 *pNewSrc, int nNewPitch, MVPlaneSet nMode)
{
   pFrames[0]->ChangePlane(pNewSrc, nNewPitch, nMode);
}

void MVGroupOfFrames::SetFrameIdx(int _nFrameIdx)
{
   nRefCount = 0;
   nFrameIdx = _nFrameIdx;
}

void MVGroupOfFrames::Refine(MVPlaneSet nMode)
{
   pFrames[0]->Refine(nMode);
}

void MVGroupOfFrames::Pad(MVPlaneSet nMode)
{
   pFrames[0]->Pad(nMode);
}

void MVGroupOfFrames::Reduce()
{
   for (int i = 0; i < nLevelCount - 1; i++ )
   {
      pFrames[i]->ReduceTo(pFrames[i+1]);
      pFrames[i+1]->Pad(YUVPLANES);
   }
}

void MVGroupOfFrames::ResetState()
{
   for ( int i = 0; i < nLevelCount; i++ )
      pFrames[i]->ResetState();
}

/******************************************************************************
*                                                                             *
*  MVFrames : manage and bufferize (group of) frames from the same clip       *
*                                                                             *
******************************************************************************/

MVFrames::MVFrames(int _nIdx, int _nNbFrames, int nLevelCount, int nWidth, int nHeight, int nPel, int nHPad, int nVPad, int nMode, bool isse, int yRatioUV)
{
   nNbFrames = _nNbFrames;
   nIdx = _nIdx;

   pGroupsOfFrames = new MVGroupOfFrames*[nNbFrames];
   for ( int i = 0; i < nNbFrames; i++ )
      pGroupsOfFrames[i] = new MVGroupOfFrames(nLevelCount, nWidth, nHeight, nPel, nHPad, nVPad, nMode, isse, yRatioUV);

}

MVFrames::~MVFrames()
{
   for ( int i = 0; i < nNbFrames; i++ )
      delete pGroupsOfFrames[i];
   delete[] pGroupsOfFrames;
}

MVGroupOfFrames *MVFrames::GetFrame(int nFrameIdx)
{
   DebugPrintf("GOF %i is requesting frame %i", nIdx, nFrameIdx);
   for ( int i = 0; i < nNbFrames; i++ )
      if ( pGroupsOfFrames[i]->GetFrameIdx() == nFrameIdx )
      {
         DebugPrintf("  --> Frame %i already found", nFrameIdx);
         pGroupsOfFrames[i]->IncRefCount();
         return pGroupsOfFrames[i];
      }

   DebugPrintf("  --> Frame %i need to be instanciated", nFrameIdx);

   return GetNewFrame(nFrameIdx);
}

MVGroupOfFrames *MVFrames::GetNewFrame(int nFrameIdx)
{
   int i;
   int maxDiff = 0, maxDiffIdx = -1;

   for ( i = 0; i < nNbFrames; i++ )
   {
      if ( pGroupsOfFrames[i]->GetFrameIdx() < 0 ) break;
      if ( abs(pGroupsOfFrames[i]->GetFrameIdx() - nFrameIdx) > maxDiff )
      {
         maxDiffIdx = i;
         maxDiff = abs(pGroupsOfFrames[i]->GetFrameIdx() - nFrameIdx);
      }
   }

   if ( i == nNbFrames )
   {
      // we have no unused space
      
      // we'll use the oldest used space
      DebugPrintf("  --> Frame %i deleted", pGroupsOfFrames[maxDiffIdx]->GetFrameIdx());
      pGroupsOfFrames[maxDiffIdx]->SetFrameIdx(nFrameIdx);
      pGroupsOfFrames[maxDiffIdx]->ResetState();
      return pGroupsOfFrames[maxDiffIdx];
   }
   else
   {
      // we have at least one unused space, so we return it
      DebugPrintf("  --> Unused space used");
      pGroupsOfFrames[i]->SetFrameIdx(nFrameIdx);
      DebugPrintf("pong\n");
      pGroupsOfFrames[i]->ResetState();
      return pGroupsOfFrames[i];
   }
}

/******************************************************************************
*                                                                             *
*  MVFramesList : manages the MVFrames of several different clips             *
*                                                                             *
******************************************************************************/

MVFramesList::MVFramesList(MVFramesList *_pNext, int nIdx, int nNbFrames, int nLevelCount, int nWidth, int nHeight, int nPel, int nHPad, int nVPad, int nMode, bool isse, int yRatioUV)
{
   pNext = _pNext;
   pMVFrames = new MVFrames(nIdx, nNbFrames, nLevelCount, nWidth, nHeight, nPel, nHPad, nVPad, nMode, isse, yRatioUV);
}
MVFramesList::MVFramesList()
{
   pNext = 0;
   pMVFrames = 0;
}

MVFramesList::~MVFramesList()
{
   if ( pNext )
      delete pNext;
   if ( pMVFrames )
      delete pMVFrames;
}

MVFrames *MVFramesList::GetFrames(int _nIdx)
{
   if ((pMVFrames) && (pMVFrames->GetIdx() == _nIdx))
      return pMVFrames;
   if (pNext)
      return pNext->GetFrames(_nIdx);
   return 0;
}

