// Define the BlockData class

// I borrowed a lot of code from XviD's sources here, so I thank all the developpers
// of this wonderful codec

// See legal notice in Copying.txt for more information

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA, or visit
// http://www.gnu.org/copyleft/gpl.html .

#ifndef __MV_INTERFACES_H__
#define __MV_INTERFACES_H__

#define _STLP_USE_STATIC_LIB

#pragma pack(16)
#pragma warning(disable:4103) // disable pack to change alignment warning ( stlport related )
#pragma warning(disable:4800) // disable warning about bool to int unefficient conversion

#include <vector>
#include <string>
#include "avisynth.h"

//#define MOTION_DEBUG          // allows to output debug information to the debug output
//#define MOTION_PROFILE        // allows to make a profiling of the motion estimation

#define MOTION_MAGIC_KEY 31415926

struct VECTOR {
	int x;
	int y;
   int sad;
};

typedef unsigned char Uint8;

enum MVPlaneSet {
   YPLANE = 1,
   UPLANE = 2,
   VPLANE = 4,
   YUPLANES = 3,
   YVPLANES = 5,
   UVPLANES = 6,
   YUVPLANES = 7
};

/*! \brief Search type : defines the algorithm used for minimizing the SAD */
enum SearchType {
	EXHAUSTIVE = 1,
	LOGARITHMIC = 2,
	ONETIME = 4,
	NSTEP = 8,
	SQUARE = 16
};

/*! \brief Macros for accessing easily frame pointers & pitch */
#define YRPLAN(a) (a)->GetReadPtr(PLANAR_Y)
#define YWPLAN(a) (a)->GetWritePtr(PLANAR_Y)
#define URPLAN(a) (a)->GetReadPtr(PLANAR_U)
#define UWPLAN(a) (a)->GetWritePtr(PLANAR_U)
#define VRPLAN(a) (a)->GetReadPtr(PLANAR_V)
#define VWPLAN(a) (a)->GetWritePtr(PLANAR_V)
#define YPITCH(a) (a)->GetPitch(PLANAR_Y)
#define UPITCH(a) (a)->GetPitch(PLANAR_U)
#define VPITCH(a) (a)->GetPitch(PLANAR_V)

/*! \brief usefull macros */
#define MAX(a,b) (((a) < (b)) ? (b) : (a))
#define MIN(a,b) (((a) > (b)) ? (b) : (a))

#define MOTION_CALC_SRC_LUMA        0x00000001
#define MOTION_CALC_REF_LUMA        0x00000002
#define MOTION_CALC_VAR             0x00000004
#define MOTION_CALC_BLOCK           0x00000008
#define MOTION_USE_MMX              0x00000010
#define MOTION_USE_ISSE             0x00000020
#define MOTION_IS_BACKWARD          0x00000040
#define MOTION_SMALLEST_PLANE       0x00000080
#define MOTION_COMPENSATE_LUMA      0x00000100
#define MOTION_COMPENSATE_CHROMA_U  0x00000200
#define MOTION_COMPENSATE_CHROMA_V  0x00000400
#define MOTION_USE_CHROMA_MOTION    0x00000800

#define MV_DEFAULT_SCD1             300
#define MV_DEFAULT_SCD2             130

#define MV_BUFFER_FRAMES 10 // named constant - Fizick

static const VECTOR zeroMV = { 0, 0, -1 };

#define inline inline

/*! \brief debug printing information */
#ifdef MOTION_DEBUG
#include <stdio.h>
#include <stdarg.h>

static inline void DebugPrintf(char *fmt, ...)
{
   va_list args;
   char buf[1024];

   va_start(args, fmt);
   vsprintf(buf, fmt, args);
   OutputDebugString(buf);
}
#else
static inline void DebugPrintf(char *fmt, ...)
{ }
#endif

// profiling
#define MOTION_PROFILE_ME              0
#define MOTION_PROFILE_LUMA_VAR        1
#define MOTION_PROFILE_COMPENSATION    2
#define MOTION_PROFILE_INTERPOLATION   3
#define MOTION_PROFILE_COUNT           4

#ifdef MOTION_PROFILE

extern int ProfTimeTable[MOTION_PROFILE_COUNT];
extern int ProfResults[MOTION_PROFILE_COUNT * 2];
extern __int64 ProfCumulatedResults[MOTION_PROFILE_COUNT * 2];

#define PROFILE_START(x) \
{ \
   __asm { rdtsc }; \
   __asm { mov [ProfTimeTable + 4 * x], eax }; \
}

#define PROFILE_STOP(x) \
{ \
   __asm { rdtsc }; \
   __asm { sub eax, [ProfTimeTable + 4 * x] }; \
   __asm { add eax, [ProfResults + 8 * x] }; \
   __asm { mov [ProfResults + 8 * x], eax }; \
   ProfResults[2 * x + 1]++; \
}

inline static void PROFILE_INIT()
{
   for ( int i = 0; i < MOTION_PROFILE_COUNT; i++ )
   {
      ProfTimeTable[i]                 = 0;
      ProfResults[2 * i]               = 0;
      ProfResults[2 * i + 1]           = 0;
      ProfCumulatedResults[2 * i]      = 0;
      ProfCumulatedResults[2 * i + 1]  = 0;
   }
}
inline static void PROFILE_CUMULATE()
{
   for ( int i = 0; i < MOTION_PROFILE_COUNT; i++ )
   {
      ProfCumulatedResults[2 * i]     += ProfResults[2 * i];
      ProfCumulatedResults[2 * i + 1] += ProfResults[2 * i + 1];
      ProfResults[2 * i]     = 0;
      ProfResults[2 * i + 1] = 0;
   }
}
inline static void PROFILE_SHOW()
{
   __int64 nTotal = 0;
   for ( int i = 0; i < MOTION_PROFILE_COUNT; i++ )
      nTotal += ProfCumulatedResults[2*i];

   DebugPrintf("ME            : %2.1f %%", (double)ProfCumulatedResults[2 * MOTION_PROFILE_ME] * 100.0f / (nTotal + 1));
   DebugPrintf("LUMA & VAR    : %2.1f %%", (double)ProfCumulatedResults[2 * MOTION_PROFILE_LUMA_VAR] * 100.0f / (nTotal + 1));
   DebugPrintf("COMPENSATION  : %2.1f %%", (double)ProfCumulatedResults[2 * MOTION_PROFILE_COMPENSATION] * 100.0f / (nTotal + 1));
   DebugPrintf("INTERPOLATION : %2.1f %%", (double)ProfCumulatedResults[2 * MOTION_PROFILE_INTERPOLATION] * 100.0f / (nTotal + 1));
}
#else
#define PROFILE_START(x)
#define PROFILE_STOP(x)
#define PROFILE_INIT()
#define PROFILE_CUMULATE()
#define PROFILE_SHOW()
#endif

class FakeBlockData {
	int x;
	int y;
	VECTOR vector;
	int nSad;
	int nLength;
   int nVariance;
   int nLuma;
   int nRefLuma;
   int nPitch;

   const unsigned char *pRef;

	inline static int SquareLength(const VECTOR& v)
	{ return v.x * v.x + v.y * v.y; }

public : 
	FakeBlockData(int _x, int _y);
	~FakeBlockData();

	void Update(const int *array, const unsigned char *ref, int pitch);

	inline int GetX() const { return x; }
	inline int GetY() const { return y; }
	inline VECTOR GetMV() const { return vector; }
	inline int GetSAD() const { return nSad; }
	inline int GetMVLength() const { return nLength; }
   inline int GetVariance() const { return nVariance; }
   inline int GetLuma() const { return nLuma; }
   inline int GetRefLuma() const { return nRefLuma; }
   inline const unsigned char *GetRef() const { return pRef; }
   inline int GetPitch() const { return nPitch; }
};

class FakePlaneOfBlocks {
    
	int nWidth_Bi;
	int nHeight_Bi;
	int nBlkX;
	int nBlkY;
	int nBlkSize;
	int nBlkCount;
	int nPel;
	int nLogPel;
	int nScale;
	int nLogScale;
	int nOverlap;

	FakeBlockData **blocks;

public :

	FakePlaneOfBlocks(int w, int h, int size, int lv, int pel, int overlap);
	~FakePlaneOfBlocks();

	void Update(const int *array, const unsigned char *pRef, int nPitch);
	bool IsSceneChange(int nTh1, int nTh2) const;

	inline bool IsInFrame(int i) const
	{
		return (( i >= 0 ) && ( i < nBlkCount ));
	}
	
	inline const FakeBlockData& operator[](const int i) const {
		return *(blocks[i]);
	}

	inline int GetBlockCount() const { return nBlkCount; }
	inline int GetReducedWidth() const { return nBlkX; }
	inline int GetReducedHeight() const { return nBlkY; }
	inline int GetWidth() const { return nWidth_Bi; }
	inline int GetHeight() const { return nHeight_Bi; }
	inline int GetScaleLevel() const { return nLogScale; }
	inline int GetEffectiveScale() const { return nScale; }
	inline int GetBlockSize() const { return nBlkSize; }
	inline int GetPel() const { return nPel; }
   inline const FakeBlockData& GetBlock(int i) const { return *(blocks[i]); }
	inline int GetOverlap() const { return nOverlap; }
};

class FakeGroupOfPlanes {
	int nLvCount_;
	bool validity;
   int nWidth_B;
   int nHeight_B;
//   int nOverlap;
   int yRatioUV_B;
	FakePlaneOfBlocks **planes;
   const unsigned char *compensatedPlane;
   const unsigned char *compensatedPlaneU;
   const unsigned char *compensatedPlaneV;
	inline static bool GetValidity(const int *array) { return (array[1] == 1); }

public :
   FakeGroupOfPlanes();
//	FakeGroupOfPlanes(int w, int h, int size, int lv, int pel);
	~FakeGroupOfPlanes();

   void Create(int _nWidth, int _nHeight, int _nBlkSize, int _nLevelCount, int _nPel, int _nOverlap, int _yRatioUV);

	void Update(const int *array);
	bool IsSceneChange(int nThSCD1, int nThSCD2) const;

	inline const FakePlaneOfBlocks& operator[](const int i) const {
		return *(planes[i]);
	}


	inline bool GetValidity() const { return validity; }
   inline const unsigned char *GetCompensatedPlane() const { return compensatedPlane; }
   inline const unsigned char *GetCompensatedPlaneU() const { return compensatedPlaneU; }
   inline const unsigned char *GetCompensatedPlaneV() const { return compensatedPlaneV; }
   inline int GetPitch() const { return nWidth_B; }
   inline int GetPitchUV() const { return nWidth_B / 2; }

	inline const FakePlaneOfBlocks& GetPlane(int i) const { return *(planes[i]); }
//	inline int GetOverlap() const { return nOverlap; }
};

class MVCore;

class MVAnalysisData
{
public:
   /*! \brief size of a block, in pixel */
   int nBlkSize;

   /*! \brief pixel refinement of the motion estimation */
   int nPel;

   /*! \brief number of level for the hierarchal search */
   int nLvCount;

   /*! \brief difference between the index of the reference and the index of the current frame */
   int nDeltaFrame;

   /*! \brief direction of the search ( forward / backward ) */
	bool isBackward;

   /*! \brief diverse flags to set up the search */
   int nFlags;

	/*! \brief Width of the frame */
	int nWidth;
	
	/*! \brief Height of the frame */
	int nHeight;

   /*! \brief MVFrames idx */
   int nIdx;

   /*! \brief Unique identifier, not very useful */
   int nMagicKey;

   int nOverlap; // overlap block size - v1.1
   int nBlkX;
   int nBlkY;
   int pixelType; // color format
   int yRatioUV;
 
   MVCore *pmvCore;

public :

   inline void SetFlags(int _nFlags) { nFlags |= _nFlags; }
   inline int GetFlags() const { return nFlags; }
   inline int GetBlkSize() const { return nBlkSize; }
   inline int GetPel() const { return nPel; }
   inline int GetLevelCount() const { return nLvCount; }
   inline int GetFramesIdx() const { return nIdx; }
   inline bool IsBackward() const { return isBackward; }
   inline int GetMagicKey() const { return nMagicKey; }
   inline int GetDeltaFrame() const { return nDeltaFrame; }
   inline int GetWidth() const { return nWidth; }
   inline int GetHeight() const { return nHeight; }
   inline bool IsChromaMotion() const { return nFlags & MOTION_USE_CHROMA_MOTION; }
   inline MVCore *GetMVCore() const { return pmvCore; }
   inline int GetOverlap() const { return nOverlap; }
   inline int GetBlkX() const { return nBlkX; }
   inline int GetBlkY() const { return nBlkY; }
   inline int GetPixelType() const { return pixelType; }
   inline int GetYRatioUV() const { return yRatioUV; }

};

class MVClip : public GenericVideoFilter, public FakeGroupOfPlanes, public MVAnalysisData
{
	/*! \brief Number of blocks horizontaly, at the first level */
//	int nBlkX;

	/*! \brief Number of blocks verticaly, at the first level */
//	int nBlkY;

	/*! \brief Number of blocks at the first level */
	int nBlkCount;

   /*! \brief Horizontal padding */
   int nHPadding;

   /*! \brief Vertical padding */
   int nVPadding;

   /*! \brief First Scene Change Detection threshold ( compared against SAD value of the block ) */
   int nSCD1;

   /*! \brief Second Scene Change Detection threshold ( compared against the number of block over the first threshold */
   int nSCD2;

//   int nOverlap;

public :
   MVClip(const PClip &vectors, int nSCD1, int nSCD2, IScriptEnvironment *env);
   ~MVClip();

   void SetVectorsNeed(bool srcluma, bool refluma, bool var,
                       bool compy, bool compu, bool compv) const;

   void Update(int n, IScriptEnvironment *env);

   // encapsulation
//   inline int GetBlkX() const { return nBlkX; }
//   inline int GetBlkY() const { return nBlkY; }
   inline int GetBlkCount() const { return nBlkCount; }
   inline int GetHPadding() const { return nHPadding; }
   inline int GetVPadding() const { return nVPadding; }
   inline int GetThSCD1() const { return nSCD1; }
   inline int GetThSCD2() const { return nSCD2; }
   inline const FakeBlockData& GetBlock(int nLevel, int nBlk) const { return GetPlane(nLevel)[nBlk]; }
   bool IsUsable(int nSCD1, int nSCD2) const;
   bool IsUsable() const { return IsUsable(nSCD1, nSCD2); }
   bool IsSceneChange() const { return FakeGroupOfPlanes::IsSceneChange(nSCD1, nSCD2); }
//   inline int GetOverlap() const { return nOverlap; }
};

class MVClipArray
{
   int size_;
   MVClip **pmvClips;

public :
   MVClipArray(const AVSValue &vectors, int nSCD1, int nSCD2, IScriptEnvironment *env);
   ~MVClipArray();
   void Update(int n, IScriptEnvironment *env);

   inline int size() { return size_; }
   inline MVClip &operator[](int i) { return *(pmvClips[i]); }

};

class MVFilter {
protected:
	/*! \brief Number of blocks horizontaly, at the first level */
	int nBlkX;

	/*! \brief Number of blocks verticaly, at the first level */
	int nBlkY;

	/*! \brief Number of blocks at the first level */
	int nBlkCount;

	/*! \brief Number of blocks at the first level */
	int nBlkSize;

   /*! \brief Horizontal padding */
   int nHPadding;

   /*! \brief Vertical padding */
   int nVPadding;

	/*! \brief Width of the frame */
	int nWidth;
	
	/*! \brief Height of the frame */
	int nHeight;

   /*! \brief MVFrames idx */
   int nIdx;

   /*! \brief pixel refinement of the motion estimation */
   int nPel;

   int nOverlap;
   int pixelType;
   int yRatioUV;

   /*! \brief Filter's name */
   std::string name;

   /*! \brief Pointer to the MVCore object */
   MVCore *mvCore;

   MVFilter(const PClip &vector, const char *filterName, IScriptEnvironment *env);

   void CheckSimilarity(const MVClip &vector, const char *vectorName, IScriptEnvironment *env);
};

#define MOTION_DELTA_FRAME_BUFFER 5

class MVPlane {
   Uint8 **pPlane;
   int nWidth;
   int nHeight;
   int nExtendedWidth;
   int nExtendedHeight;
   int nPitch;
   int nHPadding;
   int nVPadding;
   int nOffsetPadding;

   int nPel;

   bool isse;

   bool isPadded;
   bool isRefined;
   bool isFilled;

public :

   MVPlane(int _nWidth, int _nHeight, int _nPel, int _nHPad, int _nVPad, bool _isse);
   ~MVPlane();

   void ChangePlane(const Uint8 *pNewPlane, int nNewPitch);
   void Pad();
   void Refine();
   void ReduceTo(MVPlane *pReducedPlane);
   void WritePlane(FILE *pFile);

   inline const Uint8 *GetAbsolutePointer(int nX, int nY) const 
   {
      if ( nPel == 1 )
         return pPlane[0] + nX + nY * nPitch;

      int idx = (nX&1) | ((nY&1)<<1);

      nX >>= 1;
      nY >>= 1;

      return pPlane[idx] + nX + nY * nPitch;
   }

   inline const Uint8 *GetPointer(int nX, int nY) const 
   {
      return GetAbsolutePointer(nX + nHPadding * nPel, nY + nVPadding * nPel);
   }

   inline const Uint8 *GetAbsolutePelPointer(int nX, int nY) const
   {  return pPlane[0] + nX + nY * nPitch; }

   inline int GetPitch() const { return nPitch; }
   inline int GetWidth() const { return nWidth; }
   inline int GetHeight() const { return nHeight; }
   inline int GetExtendedWidth() const { return nExtendedWidth; }
   inline int GetExtendedHeight() const { return nExtendedHeight; }
   inline int GetHPadding() const { return nHPadding; }
   inline int GetVPadding() const { return nVPadding; }
   inline void ResetState() { isRefined = isFilled = isPadded = false; }

};

class MVFrame {

   MVPlane *pYPlane;
   MVPlane *pUPlane;
   MVPlane *pVPlane;

   int nMode;
   bool isse;
   int yRatioUV;

public:
   MVFrame(int nWidth, int nHeight, int nPel, int nHPad, int nVPad, int _nMode, bool _isse, int _yRatioUV);
   ~MVFrame();

   void ChangePlane(const Uint8 *pNewSrc, int nNewPitch, MVPlaneSet _nMode);
   void Refine(MVPlaneSet _nMode);
   void Pad(MVPlaneSet _nMode);
   void ReduceTo(MVFrame *pFrame);
   void ResetState();
   void WriteFrame(FILE *pFile);

   inline MVPlane *GetPlane(MVPlaneSet _nMode)
   {
      if ( nMode & _nMode & YPLANE )
         return pYPlane;

      if ( nMode & _nMode & UPLANE )
         return pUPlane;

      if ( nMode & _nMode & VPLANE )
         return pVPlane;

      return 0;
   }

   inline int GetMode() { return nMode; }

};

class MVGroupOfFrames {

   int nLevelCount;
   MVFrame **pFrames;

   int nFrameIdx;
   int nRefCount;

public :

   MVGroupOfFrames(int _nLevelCount, int nWidth, int nHeight, int nPel, int nHPad, int nVPad, int nMode, bool isse, int yRatioUV);
   ~MVGroupOfFrames();

   MVFrame *GetFrame(int nLevel);
   void SetPlane(const Uint8 *pNewSrc, int nNewPitch, MVPlaneSet nMode);
   void SetFrameIdx(int _nFrameIdx);
   void Refine(MVPlaneSet nMode);
   void Pad(MVPlaneSet nMode);
   void Reduce();
   void ResetState();

   inline int GetFrameIdx() { return nFrameIdx; }
   inline int GetRefCount() { return nRefCount; }
   inline void IncRefCount() { nRefCount++; }
};

class MVFrames {

   MVGroupOfFrames **pGroupsOfFrames;
   int nNbFrames;
   int nIdx;

public:

   MVFrames(int _nIdx, int _nNbFrames, int nLevelCount, int nWidth, int nHeight, int nPel, int nHPad, int nVPad, int nMode, bool isse, int yRatioUV);
   ~MVFrames();

   MVGroupOfFrames *GetNewFrame(int nFrameIdx);
   MVGroupOfFrames *GetFrame(int nFrameIdx);
   inline int GetIdx() { return nIdx; }

};

class MVFramesList {
   MVFrames *pMVFrames;
   MVFramesList *pNext;

public :

   MVFramesList(MVFramesList *_pNext, int nIdx, int nNbFrames, int nLevelCount, int nWidth, int nHeight, int nPel, int nHPad, int nVPad, int nMode, bool isse, int yRatioUV);
   MVFramesList();
   ~MVFramesList();

   MVFrames *GetFrames(int _nIdx);
};

class MVCore {

   MVFramesList *pFramesList;

   int nAutoIdx;

   int nRefCount;

public:

   MVCore();
   ~MVCore();

   void AddFrames(int nIdx, int nNbFrames, int nLevelCount, int nWidth,
                  int nHeight, int nPel, int nHPad, int nVPad, int nMode, bool isse, int yRatioUV); 
   MVFrames *GetFrames(int nIdx);

   inline int GetNextIdx() { nAutoIdx--; return nAutoIdx; }

};

#endif