// Change the number of frames of the clip without changing its speed

// See legal notice in Copying.txt for more information

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA, or visit
// http://www.gnu.org/copyleft/gpl.html .

#include "MVInterpolate.h"
#include <memory.h>
#include <string.h>
#include <stdio.h>

MVInterpolate::MVInterpolate(PClip _child, const VideoInfo &vi2, int nb, double rl, int fbw, int sbw, int rbw, int bvt, const char *str, const MotionParameters &params, IScriptEnvironment* env) :
GenericMotionFilter(_child, params, vi2, env)
{
	nbInterpolatedFrame = nb;
	halfWidth = nWidth / 2;
	halfHeight = nHeight / 2;
	halfBlockHeight = nBlkSize / 2;
	halfBlockWidth = nBlkSize / 2;

	revBorderWidth = rbw;
	revHalfBorderWidth = revBorderWidth / 2;
	firstBorderWidth = fbw;
	halfFirstBorderWidth = firstBorderWidth / 2;
	secondBorderWidth = sbw;
	halfSecondBorderWidth = secondBorderWidth / 2;
	borderWidth = firstBorderWidth + secondBorderWidth;
	halfBorderWidth = borderWidth / 2;
	
	blockValidityThreshold = bvt;

	relativeLength = (rl < 0 ) ? -rl : rl;
	if ( nbInterpolatedFrame < 1 ) nbInterpolatedFrame = 1;
	if ( rl < 0.001 )
	{
		rl = 0;
		nbInterpolatedFrame = 1;
	}
	if ( rl > 0.001 )
	{
		nbInterpolatedFrame = 2;
	}

	weights = new int[nbInterpolatedFrame];
	BuildWeights(str);
	BuildSmoothWeights();

	compensationsy = new int[nWidth * nHeight];
	nbcompensationsy = new int[nWidth * nHeight];
	wcompensationsy = new int[nWidth * nHeight];
	nbwcompensationsy = new int[nWidth * nHeight];
	wwcompensationsy = new int[nWidth * nHeight];
	nbwwcompensationsy = new int[nWidth * nHeight];

	compensationsu = new int[halfWidth * halfHeight];
	nbcompensationsu = new int[halfWidth * halfHeight];
	compensationsv = new int[halfWidth * halfHeight];
	nbcompensationsv = new int[halfWidth * halfHeight];

	wcompensationsu = new int[halfWidth * halfHeight];
	nbwcompensationsu = new int[halfWidth * halfHeight];
	wcompensationsv = new int[halfWidth * halfHeight];
	nbwcompensationsv = new int[halfWidth * halfHeight];

	wwcompensationsu = new int[halfWidth * halfHeight];
	nbwwcompensationsu = new int[halfWidth * halfHeight];
	wwcompensationsv = new int[halfWidth * halfHeight];
	nbwwcompensationsv = new int[halfWidth * halfHeight];
}

MVInterpolate::~MVInterpolate()
{
	delete[] compensationsy;
	delete[] nbcompensationsy;
	delete[] compensationsu;
	delete[] nbcompensationsu;
	delete[] compensationsv;
	delete[] nbcompensationsv;

	delete[] wcompensationsy;
	delete[] nbwcompensationsy;
	delete[] wcompensationsu;
	delete[] nbwcompensationsu;
	delete[] wcompensationsv;
	delete[] nbwcompensationsv;

	delete[] wwcompensationsy;
	delete[] nbwwcompensationsy;
	delete[] wwcompensationsu;
	delete[] nbwwcompensationsu;
	delete[] wwcompensationsv;
	delete[] nbwwcompensationsv;

	delete[] weights;

	for ( int i = 0; i < nBlkSize + borderWidth * 2; i++ )
		delete[] smoothWeights[i];
	delete[] smoothWeights;

	for ( int i = 0; i < halfBlockWidth + halfBorderWidth * 2; i++ )
		delete[] halfSmoothWeights[i];
	delete[] halfSmoothWeights;
}

void MVInterpolate::BuildWeights(const char *str)
{
	int i;
	if ( !lstrcmpi(str, "uniform") )
		for ( i = 0; i < nbInterpolatedFrame; i++ )
			weights[i] = 1;
	else if ( !lstrcmpi(str, "revramp") )
		for ( i = 0; i < nbInterpolatedFrame; i++ )
			weights[i] = nbInterpolatedFrame - i;
	else if ( !lstrcmpi(str, "ramp") )
		for ( i = 0; i < nbInterpolatedFrame; i++ )
			weights[i] = i + 1;
	else if ( !lstrcmpi(str, "hat") )
	{
		for ( i = 0; i < nbInterpolatedFrame / 2; i++ )
			weights[i] = i + 1;
		for ( i = nbInterpolatedFrame / 2; i < nbInterpolatedFrame; i++ )
			weights[i] = nbInterpolatedFrame - i;
	}
	else 
		for ( i = 0; i < nbInterpolatedFrame; i++ )
			weights[i] = nbInterpolatedFrame - i;
}

void MVInterpolate::BuildSmoothWeights()
{
	int totalWSize = borderWidth * 2 + nBlkSize;
	int totalHSize = borderWidth * 2 + nBlkSize;
	int halfTotalWSize = halfBorderWidth * 2 + halfBlockWidth;
	int halfTotalHSize = halfBorderWidth * 2 + halfBlockHeight;

	smoothWeights = new int*[totalWSize];
	halfSmoothWeights = new int*[halfTotalWSize];

	for ( int i = 0; i < secondBorderWidth; i++ )
	{
		smoothWeights[i] = new int[totalHSize];
		for ( int j = 0; j < secondBorderWidth; j++ )
			smoothWeights[i][j] = ( i > j ) ? j + 1 : i + 1;
		for ( int j = secondBorderWidth; j < totalHSize - secondBorderWidth; j++ )
			smoothWeights[i][j] = i + 1;
		for ( int j = totalHSize - secondBorderWidth; j < totalHSize; j++ )
			smoothWeights[i][j] = ( i > totalHSize - j - 1) ? totalHSize - j : i + 1;
	}
	for ( int i = secondBorderWidth; i < totalWSize - secondBorderWidth; i++ )
	{
		smoothWeights[i] = new int[totalHSize];
		for ( int j = 0; j < secondBorderWidth; j++ )
			smoothWeights[i][j] = j + 1;
		for ( int j = secondBorderWidth; j < totalHSize - secondBorderWidth; j++ )
			smoothWeights[i][j] = secondBorderWidth + 1;
		for ( int j = totalHSize - secondBorderWidth; j < totalHSize; j++ )
			smoothWeights[i][j] = totalHSize - j;
	}
	for ( int i = totalWSize - secondBorderWidth; i < totalWSize; i++ )
	{
		smoothWeights[i] = new int[totalHSize];
		for ( int j = 0; j < secondBorderWidth; j++ )
			smoothWeights[i][j] = ( j > totalWSize - i - 1) ? totalWSize - i : j + 1;
		for ( int j = secondBorderWidth; j < totalHSize - secondBorderWidth; j++ )
			smoothWeights[i][j] = totalWSize - i;
		for ( int j = totalHSize - secondBorderWidth; j < totalHSize; j++ )
			smoothWeights[i][j] = ( totalHSize - j - 1 > totalWSize - i - 1) ? totalWSize - i : totalHSize - j;
	}

	for ( int i = 0; i < halfSecondBorderWidth; i++ )
	{
		halfSmoothWeights[i] = new int[halfTotalHSize];
		for ( int j = 0; j < halfSecondBorderWidth; j++ )
			halfSmoothWeights[i][j] = ( i > j ) ? j + 1 : i + 1;
		for ( int j = halfSecondBorderWidth; j < halfTotalHSize - halfSecondBorderWidth; j++ )
			halfSmoothWeights[i][j] = i + 1;
		for ( int j = halfTotalHSize - halfSecondBorderWidth; j < halfTotalHSize; j++ )
			halfSmoothWeights[i][j] = ( i > halfTotalHSize - j - 1) ? halfTotalHSize - j : i + 1;
	}
	for ( int i = halfSecondBorderWidth; i < halfTotalWSize - halfSecondBorderWidth; i++ )
	{
		halfSmoothWeights[i] = new int[halfTotalHSize];
		for ( int j = 0; j < halfSecondBorderWidth; j++ )
			halfSmoothWeights[i][j] = j + 1;
		for ( int j = halfSecondBorderWidth; j < halfTotalHSize - halfSecondBorderWidth; j++ )
			halfSmoothWeights[i][j] = halfSecondBorderWidth + 1;
		for ( int j = halfTotalHSize - halfSecondBorderWidth; j < halfTotalHSize; j++ )
			halfSmoothWeights[i][j] = halfTotalHSize - j;
	}
	for ( int i = halfTotalWSize - halfSecondBorderWidth; i < halfTotalWSize; i++ )
	{
		halfSmoothWeights[i] = new int[halfTotalHSize];
		for ( int j = 0; j < halfSecondBorderWidth; j++ )
			halfSmoothWeights[i][j] = ( j > halfTotalHSize - i - 1) ? halfTotalWSize - i : j + 1;
		for ( int j = halfSecondBorderWidth; j < halfTotalHSize - halfSecondBorderWidth; j++ )
			halfSmoothWeights[i][j] = halfTotalWSize - i;
		for ( int j = halfTotalHSize - halfSecondBorderWidth; j < halfTotalHSize; j++ )
			halfSmoothWeights[i][j] = ( halfTotalHSize - j - 1> halfTotalWSize - i - 1) ? halfTotalWSize - i : halfTotalHSize - j;
	}

}


void MVInterpolate::RevTreatBlock(const FakeBlockData &block, const PVideoFrame &src, double pos)
{
	int i, j, k;

	const unsigned char *srcy = src->GetReadPtr(PLANAR_Y);
	const unsigned char *srcu = src->GetReadPtr(PLANAR_U);
	const unsigned char *srcv = src->GetReadPtr(PLANAR_V);
	const int srcpy = src->GetPitch(PLANAR_Y);
	const int srcpuv = src->GetPitch(PLANAR_U);

	int vx = -block.GetMV().x;
	int vy = -block.GetMV().y;

	int x = block.GetX();
	int y = block.GetY();

	int minx = ( x - revBorderWidth < 0 ) ? revBorderWidth - x : 0;
	int miny = ( y - revBorderWidth < 0 ) ? revBorderWidth - y : 0;
	int maxx = ( x + revBorderWidth + nBlkSize > nWidth ) ? nWidth - x + revBorderWidth : 2 * revBorderWidth + nBlkSize;
	int maxy = ( y + revBorderWidth + nBlkSize > nHeight ) ? nHeight - y + revBorderWidth : 2 * revBorderWidth + nBlkSize;

	int x2 = x / 2;
	int y2 = y / 2;

	int minx2 = ( x2 - revHalfBorderWidth < 0 ) ? revHalfBorderWidth - x2 : 0;
	int miny2 = ( y2 - revHalfBorderWidth < 0 ) ? revHalfBorderWidth - y2 : 0;
	int maxx2 = ( x2 + revHalfBorderWidth + halfBlockWidth > halfWidth ) ? halfWidth - x2 + revHalfBorderWidth : 2 * revHalfBorderWidth + halfBlockWidth;
	int maxy2 = ( y2 + revHalfBorderWidth + halfBlockHeight > halfHeight ) ? halfHeight - y2 + revHalfBorderWidth : 2 * revHalfBorderWidth + halfBlockHeight;

	double beginLength = pos - (relativeLength / 2);
	double endLength = pos + (relativeLength / 2);

	//if ( beginLength < 0 ) beginLength = 0;
	//if ( endLength > 1 ) endLength = 1.0;

	double rl = endLength - beginLength;

	int wsad = 64 * nBlkSize * nBlkSize / (block.GetSAD() + 1);

	for ( k = 0; k < nbInterpolatedFrame; k++ )
	{
		// Computation of the vector by which we'll move the block
		int vx1 = (int)((vx * ( beginLength + k * rl / nbInterpolatedFrame )) / nPel + 0.5);
		int vy1 = (int)((vy * ( beginLength + k * rl / nbInterpolatedFrame )) / nPel + 0.5);

		// We don't want indices going out of the frame

		int minxx = ( x + vx1 - revBorderWidth < 0 ) ? revBorderWidth - x - vx1 : 0;
		int minyy = ( y + vy1 - revBorderWidth < 0 ) ? revBorderWidth - y - vy1 : 0;
		int maxxx = ( x + vx1 + revBorderWidth + nBlkSize > nWidth ) ? nWidth - x - vx1 + revBorderWidth : 2 * revBorderWidth + nBlkSize;
		int maxyy = ( y + vy1 + revBorderWidth + nBlkSize > nHeight ) ? nHeight - y - vy1 + revBorderWidth : 2 * revBorderWidth + nBlkSize;

		minxx = ( minx < minxx ) ? minxx : minx;
		maxxx = ( maxx < maxxx ) ? maxx : maxxx;
		minyy = ( miny < minyy ) ? minyy : miny;
		maxyy = ( maxy < maxyy ) ? maxy : maxyy;

		int i0 = minxx;
		int i3 = maxxx;
		if ( i0 + x + vx1 >= nWidth ) i0 = i3;
		if ( i3 + x + vx1 < 0 ) i3 = i0;

		int j0 = minyy;
		int j3 = maxyy;
		if ( j0 + y + vy1 >= nHeight ) j0 = j3;
		if ( j3 + y + vy1 < 0 ) j3 = j0;

		// Pointers to the frame / newframe / utilitary planes
		int *wcy = wwcompensationsy + x + vx1 - revBorderWidth + (y + vy1 - revBorderWidth + j0) * nWidth;
		int *wny = nbwwcompensationsy + x + vx1 - revBorderWidth + (y + vy1 - revBorderWidth + j0) * nWidth;
		const unsigned char *sy = srcy + x - revBorderWidth + (y - revBorderWidth + j0) * srcpy;

		// Weight
		int wk = (int)((80.0 * PosWeight(pos) + SADWeight(block.GetSAD())) * weights[k]);

		for ( j = j0; j < j3; j++ )
		{
			for ( i = i0; i < i3; i++ )
			{
				wcy[i] += ((int)(sy[i])) * wk;
				wny[i] += wk;
			}
			wcy += nWidth;
			wny += nWidth;
			sy += srcpy;
		}

		vx1 = (int)((vx * ( beginLength + k * rl / nbInterpolatedFrame )) / ( nPel * 2 ) + 0.5);
		vy1 = (int)((vy * ( beginLength + k * rl / nbInterpolatedFrame )) / ( nPel * 2 ) + 0.5);

		minxx = ( x2 + vx1 - revHalfBorderWidth < 0 ) ? revHalfBorderWidth - x2 - vx1 : 0;
		minyy = ( y2 + vy1 - revHalfBorderWidth < 0 ) ? revHalfBorderWidth - y2 - vy1 : 0;
		maxxx = ( x2 + vx1 + revHalfBorderWidth + halfBlockWidth > halfWidth ) ? halfWidth - x2 - vx1 + revHalfBorderWidth : 2 * revHalfBorderWidth + halfBlockWidth;
		maxyy = ( y2 + vy1 + revHalfBorderWidth + halfBlockHeight > halfHeight ) ? halfHeight - y2 - vy1 + revHalfBorderWidth : 2 * revHalfBorderWidth + halfBlockHeight;

		minxx = ( minx2 < minxx ) ? minxx : minx2;
		maxxx = ( maxx2 < maxxx ) ? maxx2 : maxxx;
		minyy = ( miny2 < minyy ) ? minyy : miny2;
		maxyy = ( maxy2 < maxyy ) ? maxy2 : maxyy;

		i0 = minxx;
		i3 = maxxx;
		if ( i0 + x2 + vx1 >= halfWidth ) i0 = i3;
		if ( i3 + x2 + vx1 < 0 ) i3 = i0;

		j0 = minyy;
		j3 = maxyy;
		if ( j0 + y2 + vy1 >= halfHeight ) j0 = j3;
		if ( j3 + y2 + vy1 < 0 ) j3 = j0;

		int *cu = wwcompensationsu + x2 + vx1 - revHalfBorderWidth + (y2 + vy1 - revHalfBorderWidth + j0) * halfWidth;
		int *cv = wwcompensationsv + x2 + vx1 - revHalfBorderWidth + (y2 + vy1 - revHalfBorderWidth + j0) * halfWidth;
		int *nu = nbwwcompensationsu + x2 + vx1 - revHalfBorderWidth + (y2 + vy1 - revHalfBorderWidth + j0) * halfWidth;
		int *nv = nbwwcompensationsv + x2 + vx1 - revHalfBorderWidth + (y2 + vy1 - revHalfBorderWidth + j0) * halfWidth;
		const unsigned char *su = srcu + x2 - revHalfBorderWidth + (y2 - revHalfBorderWidth + j0) * srcpuv;
		const unsigned char *sv = srcv + x2 - revHalfBorderWidth + (y2 - revHalfBorderWidth + j0) * srcpuv;

		for ( j = j0; j < j3; j++ )
		{
			for ( i = i0; i < i3; i++ )
			{
				cu[i] += su[i] * wk;
				nu[i] += wk;
				cv[i] += sv[i] * wk;
				nv[i] += wk;
			}
			cu += halfWidth;
			nu += halfWidth;
			cv += halfWidth;
			nv += halfWidth;
			su += srcpuv;
			sv += srcpuv;
		}
	}
}

void MVInterpolate::TreatBlock(const FakeBlockData &block, const PVideoFrame &src, double pos)
{
	int i, j, k;

	const unsigned char *srcy = src->GetReadPtr(PLANAR_Y);
	const unsigned char *srcu = src->GetReadPtr(PLANAR_U);
	const unsigned char *srcv = src->GetReadPtr(PLANAR_V);
	const int srcpy = src->GetPitch(PLANAR_Y);
	const int srcpuv = src->GetPitch(PLANAR_U);

	int vx = block.GetMV().x;
	int vy = block.GetMV().y;

	int x = block.GetX();
	int y = block.GetY();

	int minx = ( x - borderWidth < 0 ) ? borderWidth - x : 0;
	int miny = ( y - borderWidth < 0 ) ? borderWidth - y : 0;
	int maxx = ( x + borderWidth + nBlkSize > nWidth ) ? nWidth - x + borderWidth : 2 * borderWidth + nBlkSize;
	int maxy = ( y + borderWidth + nBlkSize > nHeight ) ? nHeight - y + borderWidth : 2 * borderWidth + nBlkSize;

	int x2 = x / 2;
	int y2 = y / 2;

	int minx2 = ( x2 - halfBorderWidth < 0 ) ? halfBorderWidth - x2 : 0;
	int miny2 = ( y2 - halfBorderWidth < 0 ) ? halfBorderWidth - y2 : 0;
	int maxx2 = ( x2 + halfBorderWidth + halfBlockWidth > halfWidth ) ? halfWidth - x2 + halfBorderWidth : 2 * halfBorderWidth + halfBlockWidth;
	int maxy2 = ( y2 + halfBorderWidth + halfBlockHeight > halfHeight ) ? halfHeight - y2 + halfBorderWidth : 2 * halfBorderWidth + halfBlockHeight;

	double beginLength = pos - (relativeLength / 2);
	double endLength = pos + (relativeLength / 2);

//	if ( beginLength < 0 ) beginLength = 0;
//	if ( endLength > 1 ) endLength = 1.0;

	double rl = endLength - beginLength;

	int wsad = 64 * nBlkSize * nBlkSize / (block.GetSAD() + 1);

	for ( k = 0; k < nbInterpolatedFrame; k++ )
	{
		// Computation of the vector by which we'll move the block
		int vx1 = (int)(((double)vx * ( beginLength + (double)k * rl / nbInterpolatedFrame )) / (double)nPel + 0.5);
		int vy1 = (int)(((double)vy * ( beginLength + (double)k * rl / nbInterpolatedFrame )) / (double)nPel + 0.5);

		// We don't want indices going out of the frame

		int minxx = ( x + vx1 - borderWidth < 0 ) ? borderWidth - x - vx1 : 0;
		int minyy = ( y + vy1 - borderWidth < 0 ) ? borderWidth - y - vy1 : 0;
		int maxxx = ( x + vx1 + borderWidth + nBlkSize > nWidth ) ? nWidth - x - vx1 + borderWidth : 2 * borderWidth + nBlkSize;
		int maxyy = ( y + vy1 + borderWidth + nBlkSize > nHeight ) ? nHeight - y - vy1 + borderWidth : 2 * borderWidth + nBlkSize;

		minxx = ( minx < minxx ) ? minxx : minx;
		maxxx = ( maxx < maxxx ) ? maxx : maxxx;
		minyy = ( miny < minyy ) ? minyy : miny;
		maxyy = ( maxy < maxyy ) ? maxy : maxyy;

		int i1 = ( minxx < secondBorderWidth ) ? secondBorderWidth : minxx;
		int i0 = ( minxx < secondBorderWidth ) ? minxx : i1;
		int i2 = ( maxxx > nBlkSize + firstBorderWidth ) ? nBlkSize + firstBorderWidth : maxxx;
		int i3 = ( maxxx > nBlkSize + firstBorderWidth ) ? maxxx : i2;
		if ( i1 + x + vx1 >= nWidth ) i1 = i2;
		if ( i0 + x + vx1 >= nWidth ) i0 = i1;
		if ( i2 + x + vx1 < 0 ) i2 = i1;
		if ( i3 + x + vx1 < 0 ) i3 = i2;

		int j1 = ( minyy < secondBorderWidth ) ? secondBorderWidth : minyy;
		int j0 = ( minyy < secondBorderWidth ) ? minyy : j1;
		int j2 = ( maxyy > nBlkSize + firstBorderWidth ) ? nBlkSize + firstBorderWidth : maxyy;
		int j3 = ( maxyy > nBlkSize + firstBorderWidth ) ? maxyy : j2;
        if ( j1 + y + vy1 >= nHeight ) j1 = j2;
		if ( j0 + y + vy1 >= nHeight ) j0 = j1;
		if ( j2 + y + vy1 < 0 ) j2 = j1;
		if ( j3 + y + vy1 < 0 ) j3 = j2;

		// Pointers to the frame / newframe / utilitary planes
		int *cy = compensationsy + x + vx1 - borderWidth + (y + vy1 - borderWidth + j0) * nWidth;
		int *ny = nbcompensationsy + x + vx1 - borderWidth + (y + vy1 - borderWidth + j0) * nWidth;
		int *wcy = wcompensationsy + x + vx1 - borderWidth + (y + vy1 - borderWidth + j0) * nWidth;
		int *wny = nbwcompensationsy + x + vx1 - borderWidth + (y + vy1 - borderWidth + j0) * nWidth;
		const unsigned char *sy = srcy + x - borderWidth + (y - borderWidth + j0) * srcpy;

		// Weight
		int wk = (int)((80.0 * PosWeight(pos) + SADWeight(block.GetSAD())) * weights[k]);
		
		/*for ( j = j0; j < j3; j++ )
		{
			for ( i = i0; i < i3; i++ )
			{
				wcy[i] += ((int)(sy[i])) * wk * smoothWeights[i][j];
				wny[i] += wk * smoothWeights[i][j];
			}
			wcy += width;
			wny += width;
			cy += width;
			ny += width;
			sy += srcpy;
		}*/


		for ( j = j0; j < j1; j++ )
		{
			for ( i = i0; i < i3; i++ )
			{
				wcy[i] += ((int)(sy[i])) * wk * smoothWeights[i][j];
				wny[i] += wk * smoothWeights[i][j];
			}
			wcy += nWidth;
			wny += nWidth;
			cy += nWidth;
			ny += nWidth;
			sy += srcpy;
		}
		for ( j = j1; j < j2; j++ )
		{
			for ( i = i0; i < i1; i++ )
			{
				wcy[i] += ((int)(sy[i])) * wk * smoothWeights[i][j];
				wny[i] += wk * smoothWeights[i][j];
			}
			for ( i = i1; i < i2; i++ )
			{
				if ( block.GetSAD() >= blockValidityThreshold )
				{
					wcy[i] += ((int)(sy[i])) * wk * smoothWeights[i][j];
					wny[i] += wk * smoothWeights[i][j];
				}
				else {
                    cy[i] += ((int)(sy[i])) * wk;
					ny[i] += wk;
				}
			}
			for ( i = i2; i < i3; i++ )
			{
				wcy[i] += ((int)(sy[i])) * wk * smoothWeights[i][j];
				wny[i] += wk * smoothWeights[i][j];
			}
			wcy += nWidth;
			wny += nWidth;
			cy += nWidth;
			ny += nWidth;
			sy += srcpy;
		}

		for ( j = j2; j < j3; j++ )
		{
			for ( i = i0; i < i3; i++ )
			{
				wcy[i] += ((int)(sy[i])) * wk * smoothWeights[i][j];
				wny[i] += wk * smoothWeights[i][j];
			}
			wcy += nWidth;
			wny += nWidth;
			sy += srcpy;
		}

		vx1 = (int)((vx * ( beginLength + k * rl / nbInterpolatedFrame )) / ( nPel * 2 ) + 0.5);
		vy1 = (int)((vy * ( beginLength + k * rl / nbInterpolatedFrame )) / ( nPel * 2 ) + 0.5);

		minxx = ( x2 + vx1 - halfBorderWidth < 0 ) ? halfBorderWidth - x2 - vx1 : 0;
		minyy = ( y2 + vy1 - halfBorderWidth < 0 ) ? halfBorderWidth - y2 - vy1 : 0;
		maxxx = ( x2 + vx1 + halfBorderWidth + halfBlockWidth > halfWidth ) ? halfWidth - x2 - vx1 + halfBorderWidth : 2 * halfBorderWidth + halfBlockWidth;
		maxyy = ( y2 + vy1 + halfBorderWidth + halfBlockHeight > halfHeight ) ? halfHeight - y2 - vy1 + halfBorderWidth : 2 * halfBorderWidth + halfBlockHeight;

		minxx = ( minx2 < minxx ) ? minxx : minx2;
		maxxx = ( maxx2 < maxxx ) ? maxx2 : maxxx;
		minyy = ( miny2 < minyy ) ? minyy : miny2;
		maxyy = ( maxy2 < maxyy ) ? maxy2 : maxyy;

		i1 = ( minxx < halfSecondBorderWidth ) ? halfSecondBorderWidth : minxx;
		i0 = ( minxx < halfSecondBorderWidth ) ? minxx : i1;
		i2 = ( maxxx > halfBlockWidth + halfFirstBorderWidth ) ? halfBlockWidth + halfFirstBorderWidth : maxxx;
		i3 = ( maxxx > halfBlockWidth + halfFirstBorderWidth ) ? maxxx : i2;
		if ( i1 + x2 + vx1 >= halfWidth ) i1 = i2;
		if ( i0 + x2 + vx1 >= halfWidth ) i0 = i1;
		if ( i2 + x2 + vx1 < 0 ) i2 =  i1;
		if ( i3 + x2 + vx1 < 0 ) i3 =  i2;

		j1 = ( minyy < halfSecondBorderWidth ) ? halfSecondBorderWidth : minyy;
		j0 = ( minyy < halfSecondBorderWidth ) ? minyy : j1;
		j2 = ( maxyy > halfBlockHeight + halfFirstBorderWidth ) ? halfBlockHeight + halfFirstBorderWidth : maxyy;
		j3 = ( maxyy > halfBlockHeight + halfFirstBorderWidth ) ? maxyy : j2;
		if ( j1 + y2 + vy1 >= halfHeight ) j1 = j2;
		if ( j0 + y2 + vy1 >= halfHeight ) j0 = j1;
		if ( j2 + y2 + vy1 < 0 ) j2 = j1;
		if ( j3 + y2 + vy1 < 0 ) j3 = j2;

		int *cu = compensationsu + x2 + vx1 - halfBorderWidth + (y2 + vy1 - halfBorderWidth + j0) * halfWidth;
		int *cv = compensationsv + x2 + vx1 - halfBorderWidth + (y2 + vy1 - halfBorderWidth + j0) * halfWidth;
		int *nu = nbcompensationsu + x2 + vx1 - halfBorderWidth + (y2 + vy1 - halfBorderWidth + j0) * halfWidth;
		int *nv = nbcompensationsv + x2 + vx1 - halfBorderWidth + (y2 + vy1 - halfBorderWidth + j0) * halfWidth;
		const unsigned char *su = srcu + x2 - halfBorderWidth + (y2 - halfBorderWidth + j0) * srcpuv;
		const unsigned char *sv = srcv + x2 - halfBorderWidth + (y2 - halfBorderWidth + j0) * srcpuv;

		for ( j = j0; j < j3; j++ )
		{
			for ( i = i0; i < i3; i++ )
			{
				cu[i] += su[i] * wk;
				nu[i] += wk;
				cv[i] += sv[i] * wk;
				nv[i] += wk;
			}
			cu += halfWidth;
			nu += halfWidth;
			cv += halfWidth;
			nv += halfWidth;
			su += srcpuv;
			sv += srcpuv;
		}
	}
}

void MVInterpolate::BuildBlurring(PVideoFrame &dest, const PVideoFrame &src, const PVideoFrame &srcnext)
{

	unsigned char *dy = dest->GetWritePtr(PLANAR_Y);
	const unsigned char *sy = src->GetReadPtr(PLANAR_Y);
	const unsigned char *sny = srcnext->GetReadPtr(PLANAR_Y);
	const int srcpy = src->GetPitch(PLANAR_Y);
	const int dstpy = dest->GetPitch(PLANAR_Y);
	int *cy = compensationsy;
	int *ny = nbcompensationsy;
	int *wcy = wcompensationsy;
	int *wny = nbwcompensationsy;
	int *wwcy = wwcompensationsy;
	int *wwny = nbwwcompensationsy;

	for (int j = 0; j < nHeight; j++ )
	{
		for (int i = 0; i < nWidth; i++ )
		{
			dy[i] = ( ny[i] == 0 ) ? 
				( wny[i] == 0 ) ? 
					( wwny[i] == 0 ) ?
						sy[i] 
						: (wwcy[i] + (wwny[i] / 2)) / wwny[i]
					: (wcy[i] + (wny[i] / 2)) / wny[i]
				: (cy[i] + (ny[i] / 2)) / ny[i];
		}
		dy += dstpy;
		cy += nWidth;
		ny += nWidth;
		wcy += nWidth;
		wny += nWidth;
		wwcy += nWidth;
		wwny += nWidth;
		sy += srcpy;
		sny += srcpy;
	}

	unsigned char *du = dest->GetWritePtr(PLANAR_U);
	const unsigned char *su = src->GetReadPtr(PLANAR_U);
	unsigned char *dv = dest->GetWritePtr(PLANAR_V);
	const unsigned char *sv = src->GetReadPtr(PLANAR_V);
	const int srcpuv = src->GetPitch(PLANAR_U);
	const int dstpuv = dest->GetPitch(PLANAR_U);
	int *cu = compensationsu;
	int *nu = nbcompensationsu;
	int *cv = compensationsv;
	int *nv = nbcompensationsv;
	int *wwcu = wwcompensationsu;
	int *wwnu = nbwwcompensationsu;
	int *wwcv = wwcompensationsv;
	int *wwnv = nbwwcompensationsv;

	for (int j = 0; j < halfHeight; j++ )
	{
		for (int i = 0; i < halfWidth; i++ )
		{
			du[i] = ( nu[i] == 0 ) ?
				( wwnu[i] == 0 ) ?
					su[i]
					: (wwcu[i] + (wwnu[i] / 2)) / wwnu[i]
				: (cu[i] + (nu[i] / 2)) / nu[i];
			dv[i] = ( nv[i] == 0 ) ?
				( wwnv[i] == 0 ) ?
					sv[i]
					: (wwcv[i] + (wwnv[i] / 2)) / wwnv[i]
				: (cv[i] + (nv[i] / 2)) / nv[i];
		}
		cu += halfWidth;
		cv += halfWidth;
		nu += halfWidth;
		nv += halfWidth;
		wwcu += halfWidth;
		wwcv += halfWidth;
		wwnu += halfWidth;
		wwnv += halfWidth;
		su += srcpuv;
		sv += srcpuv;
		du += dstpuv;
		dv += dstpuv;
	}
}