// Show the motion vectors of a clip

// See legal notice in Copying.txt for more information

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA, or visit
// http://www.gnu.org/copyleft/gpl.html .

#include "MVShow.h"
#include <stdio.h>
#include "info.h"

MVShow::MVShow(PClip _child, PClip vectors, int _scale, int _plane, int _tol, bool _showsad,
               int nSCD1, int nSCD2, bool mmx, bool _isse, IScriptEnvironment* env) :
GenericVideoFilter(_child),
mvClip(vectors, nSCD1, nSCD2, env),
MVFilter(vectors, "MVShow", env)
{
	isse = _isse;
	nScale = _scale;
	if ( nScale < 1 )
      nScale = 1;

	nPlane = _plane;
	if ( nPlane < 0 )
      nPlane = 0;
   if ( nPlane >= mvClip.GetLevelCount() - 1 )
      nPlane = mvClip.GetLevelCount() - 1;

	nTolerance = _tol;
   if ( nTolerance < 0 )
      nTolerance = 0;

	showSad = _showsad;

	if ( (pixelType & VideoInfo::CS_YUY2) == VideoInfo::CS_YUY2 )
   {
		SrcPlanes =  new YUY2Planes(nWidth, nHeight);
		DstPlanes =  new YUY2Planes(nWidth, nHeight);
   }
}

MVShow::~MVShow()
{
   if ( (pixelType & VideoInfo::CS_YUY2) == VideoInfo::CS_YUY2 )
   {
	delete DstPlanes;
   }
}

PVideoFrame __stdcall MVShow::GetFrame(int n, IScriptEnvironment* env)
{
	PVideoFrame	src = child->GetFrame(n, env);
	PVideoFrame	dst = env->NewVideoFrame(vi);
   BYTE *pDst[3];
	   const BYTE *pSrc[3];
    int nDstPitches[3], nSrcPitches[3];
	unsigned char *pDstYUY2;
	int nDstPitchYUY2;

   mvClip.Update(n, env);

   		if ( (pixelType & VideoInfo::CS_YUY2) == VideoInfo::CS_YUY2 )
		{
			const unsigned char *pSrcYUY2 = src->GetReadPtr();
			int nSrcPitchYUY2 = src->GetPitch();
			pSrc[0] = SrcPlanes->GetPtr();
			pSrc[1] = SrcPlanes->GetPtrU();
			pSrc[2] = SrcPlanes->GetPtrV();
			nSrcPitches[0]  = SrcPlanes->GetPitch();
			nSrcPitches[1]  = SrcPlanes->GetPitchUV();
			nSrcPitches[2]  = SrcPlanes->GetPitchUV();
			YUY2ToPlanes(pSrcYUY2, nSrcPitchYUY2, nWidth, nHeight, 
				pSrc[0], nSrcPitches[0], pSrc[1], pSrc[2], nSrcPitches[1], isse);

			pDstYUY2 = dst->GetWritePtr();
			nDstPitchYUY2 = dst->GetPitch();
			pDst[0] = DstPlanes->GetPtr();
			pDst[1] = DstPlanes->GetPtrU();
			pDst[2] = DstPlanes->GetPtrV();
			nDstPitches[0]  = DstPlanes->GetPitch();
			nDstPitches[1]  = DstPlanes->GetPitchUV();
			nDstPitches[2]  = DstPlanes->GetPitchUV();
			YUY2ToPlanes(pDstYUY2, nDstPitchYUY2, nWidth, nHeight, 
				pDst[0], nDstPitches[0], pDst[1], pDst[2], nDstPitches[1], isse);
		}
		else
		{
         pSrc[0] = YRPLAN(src);
         pSrc[1] = URPLAN(src);
         pSrc[2] = VRPLAN(src);
         nSrcPitches[0] = YPITCH(src);
         nSrcPitches[1] = UPITCH(src);
         nSrcPitches[2] = VPITCH(src);

		 pDst[0] = YWPLAN(dst);
         pDst[1] = UWPLAN(dst);
         pDst[2] = VWPLAN(dst);
         nDstPitches[0] = YPITCH(dst);
         nDstPitches[1] = UPITCH(dst);
         nDstPitches[2] = VPITCH(dst);
		}

	// Copy the frame into the created frame
   env->BitBlt(pDst[1], nDstPitches[1], pSrc[1], nSrcPitches[1], nWidth >> 1, nHeight /yRatioUV);
   env->BitBlt(pDst[2], nDstPitches[2], pSrc[2], nSrcPitches[2], nWidth >> 1, nHeight /yRatioUV);
   if ( !nPlane )
	   env->BitBlt(pDst[0], nDstPitches[0], pSrc[0], nSrcPitches[0], nWidth, nHeight);

   if ( mvClip.IsUsable() )
      DrawMVs(pDst[0], nDstPitches[0], pSrc[0], nSrcPitches[0]);

		if ( (pixelType & VideoInfo::CS_YUY2) == VideoInfo::CS_YUY2 )
		{
			YUY2FromPlanes(pDstYUY2, nDstPitchYUY2, nWidth, nHeight,
								  pDst[0], nDstPitches[0], pDst[1], pDst[2], nDstPitches[1], isse);
		}

	if ( showSad ) {
		char buf[80];
		double mean = 0;
      for ( int i = 0; i < mvClip.GetBlkCount(); i++ )
         mean += mvClip.GetBlock(0, i).GetSAD();
		sprintf(buf, "%f", mean / nBlkCount);
		if ( (pixelType & VideoInfo::CS_YUY2) == VideoInfo::CS_YUY2 )
			DrawStringYUY2(dst, 0, 0, buf);
		else
			DrawString(dst, 0, 0, buf);
	}
	return dst;
}

void MVShow::DrawPixel(unsigned char *pDst, int nDstPitch, int x, int y, int w, int h)
{
	if (( x >= 0 ) && ( x < w ) && ( y >= 0 ) && ( y < h ))
		pDst[x + y * nDstPitch] = 255;
}

// Draw the vector, scaled with the right scalar factor.
void MVShow::DrawMV(unsigned char *pDst, int nDstPitch, int scale,
			        int x, int y, int size, int w, int h, VECTOR vector, int pel)
{
	int x0 = (size / 2) + x - 1;
	int y0 = (size / 2) + y - 1;

   bool yLonger = false;
	int incrementVal, endVal;
	int shortLen =  scale*vector.y / pel; // added missed scale - Fizick
	int longLen = scale*vector.x / pel;   //
	if ( abs(shortLen) > abs(longLen)) {
		int swap = shortLen;
		shortLen = longLen;
		longLen = swap;
		yLonger = true;
	}
	endVal = longLen;
	if (longLen < 0) {
		incrementVal = -1;
		longLen = -longLen;
	} else incrementVal = 1;
	int decInc;
	if (longLen==0) decInc=0;
	else decInc = (shortLen << 16) / longLen;
	int j=0;
	if (yLonger) {	
		for (int i = 0; i != endVal; i += incrementVal) {
            DrawPixel(pDst, nDstPitch, x0 + (j >> 16), y0 + i, w, h);
			j += decInc;
		}
	} else {
		for (int i = 0; i != endVal; i += incrementVal) {
            DrawPixel(pDst, nDstPitch, x0 + i, y0 + (j>> 16), w, h);
			j += decInc;
		}
	}
}

void MVShow::DrawMVs(unsigned char *pDst, int nDstPitch, const unsigned char *pSrc,
                     int nSrcPitch)
{
	const FakePlaneOfBlocks &plane = mvClip.GetPlane(nPlane);
	int effectiveScale = plane.GetEffectiveScale();

	const unsigned char *s = pSrc;
	
    if ( nPlane )
    {
	    for ( int j = 0; j < plane.GetHeight(); j++)
	    {
		    for ( int i = 0; i < plane.GetWidth(); i++)
		    {
			    int val = 0;
			    for ( int x = i * effectiveScale ; x < effectiveScale * (i + 1); x++ )
				    for ( int y = j * effectiveScale; y < effectiveScale * (j + 1); y++ )
					    val += pSrc[x + y * nSrcPitch];
			    val = (val + effectiveScale * effectiveScale / 2) / (effectiveScale * effectiveScale);
			    for ( int x = i * effectiveScale ; x < effectiveScale * (i + 1); x++ )
				    for ( int y = j * effectiveScale; y < effectiveScale * (j + 1); y++ )
					    pDst[x + y * nDstPitch] = val;
		    }
		    s += effectiveScale * nSrcPitch;
	    }
    }
	for ( int i = 0; i < plane.GetBlockCount(); i++ )
		if ( plane[i].GetSAD() < nTolerance )
			DrawMV(pDst, nDstPitch, nScale * effectiveScale, plane[i].GetX() * effectiveScale, 
				plane[i].GetY() * effectiveScale, (plane.GetBlockSize() - plane.GetOverlap())* effectiveScale,
                plane.GetWidth() * effectiveScale, plane.GetHeight() * effectiveScale, 
                plane[i].GetMV(), plane.GetPel());
}