// Functions that computes distances between blocks

// See legal notice in Copying.txt for more information

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA, or visit
// http://www.gnu.org/copyleft/gpl.html .

/*! \file SADFunctions.h
 *  \brief Calculate the sum of absolute differences between two blocks.
 * 
 *	The sum of absolute differences between two blocks represents the distance
 *	between these two blocks. The lower this value, the closer the blocks will look like
 *  There are two versions, one in plain C, one in iSSE assembler.
 */

#ifndef __SAD_FUNC__
#define __SAD_FUNC__

#include "MVInterface.h"

typedef unsigned int (SADFunction)(const BYTE *pSrc, const BYTE *pRef,
								           int nSrcPitch, int nRefPitch);

inline unsigned int SADABS(int x) {	return ( x < 0 ) ? -x : x; }

template<int nBlkWidth, int nBlkHeight>
unsigned int Sad_C(const BYTE *pSrc, const BYTE *pRef,
					    int nSrcPitch, int nRefPitch)
{
	unsigned int sum = 0;
	for ( int y = 0; y < nBlkHeight; y++ )
	{
		for ( int x = 0; x < nBlkWidth; x++ )
			sum += SADABS(pSrc[x] - pRef[x]);
      pSrc += nSrcPitch;
      pRef += nRefPitch;
	}
	return sum;
}

template<int nBlkSize>
unsigned int Sad_C(const BYTE *pSrc, const BYTE *pRef,
                   int nSrcPitch, int nRefPitch)
{
   return Sad_C<nBlkSize, nBlkSize>(pSrc, pRef, nSrcPitch, nRefPitch);
}

extern "C" unsigned int __cdecl Sad16x16_iSSE(const BYTE *pSrc, const BYTE *pRef,
									                   int nSrcPitch, int nRefPitch);
extern "C" unsigned int __cdecl Sad8x8_iSSE(const BYTE *pSrc, const BYTE *pRef,
									                 int nSrcPitch, int nRefPitch);
extern "C" unsigned int __cdecl Sad4x4_iSSE(const BYTE *pSrc, const BYTE *pRef,
									                 int nSrcPitch, int nRefPitch);
extern "C" unsigned int __cdecl Sad4x8_iSSE(const BYTE *pSrc, const BYTE *pRef,
									                 int nSrcPitch, int nRefPitch);
extern "C" unsigned int __cdecl Sad8x16_iSSE(const BYTE *pSrc, const BYTE *pRef,
									                 int nSrcPitch, int nRefPitch);


#endif