// Define the BlockData class

// I borrowed a lot of code from XviD's sources here, so I thank all the developpers
// of this wonderful codec

// See legal notice in Copying.txt for more information

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA, or visit
// http://www.gnu.org/copyleft/gpl.html .

#include "BlockData.h"

BlockData::BlockData(int x_pos, int y_pos, int w, int h, int bw, int bh, int pro, int mxd, int l, bool isMMX, bool isiSSE, PlaneOfBlocks *p)
{
	plane = p;

	x = x_pos;
	y = y_pos;

	int width = plane->width;
	int	height = plane->height;

	int	blockWidth = plane->blockWidth;
	int blockHeight = plane->blockHeight;

	int pelAccuracy = plane->pelAccuracy;

	// Calculation of the min and max value of the coordinates of the motion vector
	// of the block. We do not want him to point outside the frame.
	max_dx =  pelAccuracy * (width - x - blockWidth) + pelAccuracy;
	max_dy = pelAccuracy * (height - y - blockHeight) + pelAccuracy;
	min_dx = -pelAccuracy * x;
	min_dy = -pelAccuracy * y;
}

BlockData::~BlockData()
{
	//delete[] interpolatedFrames;
	delete[] neighbours;
	//delete[] predictors;
	delete[] previousNeighbours;
	delete[] currentNeighbours;
}

void BlockData::Initialize(int p, unsigned char * const *intf)
{
	//pitch = p;

	offset = x + y * plane->pitch;

	// This array points directly to the corresponding array in PlaneOfBlocks. This
	// association is done once and for all.
//	for ( int i = 1; i < pelRefiningOrder * pelRefiningOrder; i++ )
//		interpolatedFrames[i] = intf[i] + offset;

	predictorsCount = 2 + previousNeighboursCount + currentNeighboursCount;

	//predictors = new VECTOR[predictorsCount];
}

void BlockData::SetReferences(const unsigned char *s, const unsigned char *sp)
{
	// These two references change for each frames.
	blockReference = s + offset;
	//interpolatedFrames[0] = sp + offset;
}

// We compute the deviation of the motion vector from its surrounding. Allow
// us to know if the block can be motion compensated. If the deviation is
// too large, that means the block is locally erratic, and using it to motion 
// compensation may be hazardous.

//// Should be useless on this side of the ME
void BlockData::ComputeDifferenceFromNeighbours()
{
	// MEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFf
	//differenceFromNeighbours = 0; 
	//for ( int i = 0; i < neighboursCount; i++ )
	//	differenceFromNeighbours += SquareDifferenceNorm(finalVector, neighbours[i]->GetFinalMV());
	//differenceFromNeighbours /= neighboursCount;
}

// Go fetch the spatial predictors, the same one as in EPZ.

void BlockData::FetchCurrentPredictors()
{
	VECTOR *predictors = plane->blockPredictors;

	predictors[0] = zeroMV;
	for ( int i = 0; i < currentNeighboursCount; i++ )
	{
		predictors[i+1] = currentNeighbours[i]->GetFinalMV();
	}
	if ( currentNeighboursCount == 3 )
	{
		medianPredictor.x = MedianThree(predictors[1].x, predictors[2].x, predictors[3].x);
		medianPredictor.y = MedianThree(predictors[1].y, predictors[2].y, predictors[3].y);
	}
	else if ( currentNeighboursCount == 2 )
	{
		medianPredictor.x = (predictors[1].x + predictors[2].x + 1) / 2;
		medianPredictor.y = (predictors[1].y + predictors[2].y + 1) / 2;
	}
	else if ( currentNeighboursCount == 2 )
	{
		medianPredictor.x = predictors[1].x;
		medianPredictor.y = predictors[1].y;
	}
	else medianPredictor = zeroMV;

	predictors[currentNeighboursCount+1] = medianPredictor;
}

// Fetch the previous predictors 

void BlockData::FetchPreviousPredictors()
{
	//predictors[1 + currentNeighboursCount] = finalVector;
	//for ( int i = 0; i < previousNeighboursCount; i++ )
	//{
	//	predictors[i + 2 + currentNeighboursCount] = previousNeighbours[i]->GetFinalMV();
	//}
}

// Search the MV of the block.

void BlockData::SearchMV(VECTOR *initMV, int externalPredictorsCount, int step, SearchType st, PredictionType pt)
{
	int i;
	VECTOR *predictors = plane->blockPredictors;

	hierarchalPredictor = initMV[0]; //predictors[currentNeighboursCount+1];

	currentVector = zeroMV;
	currentSAD = 65536;

	for ( i = 0; i < externalPredictorsCount; i++ )
		CheckMV(initMV[i].x, initMV[i].y);

	if ( pt & CURRENT )
		for ( i = 0; i < currentNeighboursCount + 2; i++ )
			CheckMV(predictors[i].x, predictors[i].y );

	if ( pt & PREVIOUS )
		for ( i = 2 + currentNeighboursCount; i < predictorsCount; i++ )
			CheckMV(predictors[i].x, predictors[i].y );

	if ( st & EXHAUSTIVE )
		ExhaustiveSearch(currentVector, step * plane->pelAccuracy);

	if ( st & LOGARITHMIC )
		for ( i = step * plane->pelAccuracy; i > 0; i /= 2 )
			DiamondSearch(i);

	if ( st & ONETIME )
		for ( i = step * plane->pelAccuracy; i > 0; i /= 2 )
			OneTimeSearch(currentVector,i);

	if ( st & NSTEP )
		NStepSearch(currentVector, step * plane->pelAccuracy);
	
	if ( st & SQUARE )
		SquareSearch(currentVector);

	//bestPredictor = initpredictors[currentNeighboursCount+1];
	
}

void BlockData::PostProcessing(int threshold)
{
	finalVector = currentVector;
	finalVectorLength = SquareLength(finalVector);
	finalSAD = currentSAD;
}

void BlockData::ExhaustiveSearch(VECTOR mv, int s)
{
	int i, j;

	for ( i = -s + 1; i < 0; i++ )
		for ( j = -s + 1; j < s; j++ )
			CheckMV(mv.x + i, mv.y + j);

	for ( i = 1; i < s; i++ )
		for ( j = -s + 1; j < s; j++ )
			CheckMV(mv.x + i, mv.y + j);

	for ( j = -s + 1; j < 0; j++ )
		CheckMV(mv.x, mv.y + j);

	for ( j = 1; j < s; j++ )
		CheckMV(mv.x, mv.y + j);

}

void BlockData::SquareSearch(VECTOR mv)
{
	ExhaustiveSearch(mv,1);
}

void BlockData::OneTimeSearch(VECTOR mv, int length)
{
	int direction = 0;
	int dx = mv.x;
	int dy = mv.y;
	
	CheckMV2(dx - length, dy, &direction, 2);
	CheckMV2(dx + length, dy, &direction, 1);

	if ( direction == 1 )
	{
		while ( direction )
		{
			direction = 0;
			dx += length;
			CheckMV2(dx + length, dy, &direction, 1);
		}
	}
	else if ( direction == 2 )
	{
		while ( direction )
		{
			direction = 0;
			dx -= length;
			CheckMV2(dx - length, dy, &direction, 1);
		}
	}

	CheckMV2(dx, dy - length, &direction, 2);
	CheckMV2(dx, dy + length, &direction, 1);

	if ( direction == 1 )
	{
		while ( direction )
		{
			direction = 0;
			dy += length;
			CheckMV2(dx, dy + length, &direction, 1);
		}
	}
	else if ( direction == 2 )
	{
		while ( direction )
		{
			direction = 0;
			dy -= length;
			CheckMV2(dx, dy - length, &direction, 1);
		}
	}
}

void BlockData::NStepSearch(VECTOR mv, int length)
{
	int dx, dy;
	while ( length > 0 )
	{
		dx = currentVector.x;
		dy = currentVector.y;

		CheckMV(dx + length, dx + length);
		CheckMV(dx + length, dx);
		CheckMV(dx + length, dx - length);
		CheckMV(dx, dx - length);
		CheckMV(dx, dx + length);
		CheckMV(dx - length, dx + length);
		CheckMV(dx - length, dx);
		CheckMV(dx - length, dx - length);

		length--;
	}
}

// If you look at XviD's code, you'll recognize its spirit :)
void BlockData::DiamondSearch(int length)
{
	// The meaning of the directions are the following :
	//		* 1 means right
	//		* 2 means left
	//		* 4 means down
	//		* 8 means up
	// So 1 + 4 means down right, and so on...

	int dx;
	int dy;

	// We begin by making no assumption on which direction to search.
	int direction = 15;

	int lastDirection;

	while ( direction > 0 )
	{
		dx = currentVector.x;
		dy = currentVector.y;
		lastDirection = direction;
		direction = 0;

		// First, we look the directions that were hinted by the previous step
		// of the algorithm. If we find one, we add it to the set of directions
		// we'll test next
		if ( lastDirection & 1 ) CheckMV2(dx + length, dy, &direction, 1);
		if ( lastDirection & 2 ) CheckMV2(dx - length, dy, &direction, 2);
		if ( lastDirection & 4 ) CheckMV2(dx, dy + length, &direction, 4);
		if ( lastDirection & 8 ) CheckMV2(dx, dy - length, &direction, 8);

		// If one of the directions improves the SAD, we make further tests 
		// on the diagonals
		if ( direction ) {
			lastDirection = direction;
			dx = currentVector.x;
			dy = currentVector.y;

			if ( lastDirection & 3 ) 
			{
				CheckMV2(dx, dy + length, &direction, 4);
				CheckMV2(dx, dy - length, &direction, 8);
			}
			else {
				CheckMV2(dx + length, dy, &direction, 1);
				CheckMV2(dx - length, dy, &direction, 2);
			}
		}

		// If not, we do not stop here. We infer from the last direction the
		// diagonals to be checked, because we might be lucky.
		else {
			switch ( lastDirection ) {
				case 1 : 
					CheckMV2(dx + length, dy + length, &direction, 1 + 4);
					CheckMV2(dx + length, dy - length, &direction, 1 + 8);
					break;
				case 2 : 
					CheckMV2(dx - length, dy + length, &direction, 2 + 4);
					CheckMV2(dx - length, dy - length, &direction, 2 + 8);
					break;
				case 4 : 
					CheckMV2(dx + length, dy + length, &direction, 1 + 4);
					CheckMV2(dx - length, dy + length, &direction, 2 + 4);
					break;
				case 8 : 
					CheckMV2(dx + length, dy - length, &direction, 1 + 8);
					CheckMV2(dx - length, dy - length, &direction, 2 + 8);
					break;
				case 1 + 4 :
					CheckMV2(dx + length, dy + length, &direction, 1 + 4);
					CheckMV2(dx - length, dy + length, &direction, 2 + 4);
					CheckMV2(dx + length, dy - length, &direction, 1 + 8);
					break;
				case 2 + 4 :
					CheckMV2(dx + length, dy + length, &direction, 1 + 4);
					CheckMV2(dx - length, dy + length, &direction, 2 + 4);
					CheckMV2(dx - length, dy - length, &direction, 2 + 8);
					break;
				case 1 + 8 :
					CheckMV2(dx + length, dy + length, &direction, 1 + 4);
					CheckMV2(dx - length, dy - length, &direction, 2 + 8);
					CheckMV2(dx + length, dy - length, &direction, 1 + 8);
					break;
				case 2 + 8 :
					CheckMV2(dx - length, dy - length, &direction, 2 + 8);
					CheckMV2(dx - length, dy + length, &direction, 2 + 4);
					CheckMV2(dx + length, dy - length, &direction, 1 + 8);
					break;
				default :
					// Even the default case may happen, in the first step of the 
					// algorithm for example.
					CheckMV2(dx + length, dy + length, &direction, 1 + 4);
					CheckMV2(dx - length, dy + length, &direction, 2 + 4);
					CheckMV2(dx + length, dy - length, &direction, 1 + 8);
					CheckMV2(dx - length, dy - length, &direction, 2 + 8);
					break;
			}
		}
	}
}

void BlockData::ComputeSAD()
{
	char index = BDMod(finalVector.x) + plane->pelAccuracy * BDMod(finalVector.y);
	finalSAD = plane->SAD(blockReference, plane->interpolatedPreviousFrame[index] + BDDiv(finalVector.x) + BDDiv(finalVector.y) * plane->pitch, plane->pitch, plane->pitch);
}

void BlockData::ReadVectorFromFile(MVFileHandler *input, MVVectorsStats *mvstats, int *predx, int *predy)
{
	finalVector.x = mvstats->ReadCode(input) + *predx;
	finalVector.y = mvstats->ReadCode(input) + *predy;
	*predx = finalVector.x;
	*predy = finalVector.y;
	ComputeSAD();
}

void BlockData::WriteVectorFromFile(MVFileHandler *output, MVVectorsStats *mvstats, int *predx, int *predy)
{
	mvstats->WriteCode(output, finalVector.x - hierarchalPredictor.x);
	mvstats->WriteCode(output, finalVector.y - hierarchalPredictor.y);
	*predx = finalVector.x;
	*predy = finalVector.y;
}



		
