// Define the BlockData class

// I borrowed a lot of code from XviD's sources here, so I thank all the developpers
// of this wonderful codec

// See legal notice in Copying.txt for more information

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA, or visit
// http://www.gnu.org/copyleft/gpl.html .

#ifndef __MV__RES__
#define __MV__RES__

#include "portability.h"
//#include "SADFunctions.h"
#include "MVVectorsStats.h"
#include "PlaneOfBlocks.h"

/*! \brief Class containing all the tools in order to compute the motion vector of a block
 *
 *	A BlockData represent a block of the frame. It contains references to the current
 *  frame, and to the previous frame and all its interpolations / refinements.
 *  It has all the necessary methods in order to compute the
 *	motion vector of this block, plus other handy functions.
 */
class BlockData {

	// We want PlaneOfBlocks to access freely the fields of the BlockDatas it contains.
	// Not OO, but code inside PlaneOfBlocks is then easier to read. 
	friend class PlaneOfBlocks;

	PlaneOfBlocks *plane;

	/*! \brief Maximim admissible horizontal coordinate for the motion vector. */
	int max_dx;
	
	/*! \brief Maximim admissible vertical coordinate for the motion vector. */
	int max_dy;

	/*! \brief Minimum admissible horizontal coordinate for the motion vector. */
	int min_dx;
	
	/*! \brief Minimum admissible vertical coordinate for the motion vector. */
	int min_dy;

	/*! \brief Total number of neighbours. */
	int neighboursCount;

	/*! \brief Total number of 'current' neighbours. */
	int currentNeighboursCount;

	/*! \brief Total number of 'previous' neighbours. */
	int previousNeighboursCount;

	/*! \brief Number of first search candidates. */
	int predictorsCount;

	/*! \brief Offset of the upper left pixel of the block inside the frame. */
	unsigned int offset;

	/*! \brief Reference on the block in the current frame. */
	const unsigned char *blockReference;

	/*! \brief Array of the surrounding blocks. */
	BlockData **neighbours;

	/*! \brief Array of the surrounding blocks that are computed before the block
		and whose motion vectors are used as predictors for this block. */
	BlockData **currentNeighbours;

	/*! \brief Array of the surrounding blocks whose motion vectors computed at
		the previous instant are used are predictors for this block. */
	BlockData **previousNeighbours;

	/*! \brief Current vector which is used during computation of the motion vector. */
	VECTOR currentVector;
	VECTOR finalVector;

	/*! \brief Length of this vector. Not necessarily used. */
	unsigned int currentVectorLength;
	unsigned int finalVectorLength;

	/*! \brief Final distorsion linked to the motion vector chosen. */
	unsigned int finalSAD;
	unsigned int currentSAD;

	VECTOR hierarchalPredictor;
    
	VECTOR medianPredictor;



	///*! \brief Width of the frame. */
	//int width;
	//
	///*! \brief Height of the frame. */
	//int height;

	///*! \brief Width of the block. */
	//int blockWidth;
	//
	///*! \brief Height of the block. */
	//int blockHeight;

	///*! \brief Precision of the interpolation made before searching the motion vector. */
	//int pelRefiningOrder;


	///*! \brief Reference on the offset interpolations of the previous frame. */
	//const unsigned char **interpolatedFrames;

	///*! \brief First search candidates for the search of a motion vector. */
	//VECTOR *predictors;

	///*! \brief Sum of quadratic differences between the motion vector of the block
	// *	and all its surrounding blocks'	motion vectors.
	// */
	//int differenceFromNeighbours;

	///*! \brief Threshold under which the SAD is found good enough not to keep on the search */
	//unsigned int SADThreshold;

	//SADFunction *SAD;

	//int lambda;

	/*
		Inlined functions
	*/

	/*! \brief Computes a true algebraic modulo by pelRefiningOrder.
	 *	\param x Integer which will be modulated by pelRefiningOrder.
	 */
	___INLINE___ unsigned int BDMod(int x)
	{ return  (x < 0) ? plane->pelAccuracy - 1 + ( ( x + 1 ) % plane->pelAccuracy): (x % plane->pelAccuracy); }

	/*!	\brief Computes a true algebraic division by pelRefiningOrder.
	 *	\param x Integer to be divide by pelRefiningOrder.
	 */
	___INLINE___ int BDDiv(int x)
	{ return (x < 0) ? (( x + 1 ) / plane->pelAccuracy ) - 1: (x / plane->pelAccuracy); }

	/*! \brief Computes the quadratic difference between two vectors.
	 *	\param v1 First vector.
	 *	\param v2 Second vector.
	 */
	___INLINE___ static unsigned int SquareDifferenceNorm(const VECTOR& v1, const VECTOR& v2) 
	{ return (v1.x - v2.x) * (v1.x - v2.x) + (v1.y - v2.y) * (v1.y - v2.y); }

	___INLINE___ static unsigned int SquareDifferenceNorm(const VECTOR& v1, const int v2x, const int v2y) 
	{ return (v1.x - v2x) * (v1.x - v2x) + (v1.y - v2y) * (v1.y - v2y); }

	___INLINE___ static unsigned int SquareLength(const VECTOR& v)
	{ return v.x * v.x + v.y * v.y; }

	___INLINE___ int MedianThree(int a, int b, int c)
	{
		if ( a < b )
		{
			if ( b < c ) return b;
			else if ( a < c ) return c;
			else return a;
		}
		else {
			if ( a < c ) return a;
			else if ( b < c ) return c;
			else return b;
		}
	}

	___INLINE___ unsigned int MIN(unsigned int a, unsigned int b)
	{
		return ( a < b ) ? a : b;
	}

	/*! \brief Test of a potential motion vector
	 *
	 *	\param dx X coordinate of the tested vector.
	 *	\param dy Y coordinate of the tested vector.
	 *	\param dir 
	 *	\param val Directions which will be tested if the vector is proven to be good.
	 */
	___INLINE___ void CheckMV2(int dx, int dy, int *dir, int val)
	{
		if (( currentSAD >= plane->SADThreshold ) && ( dx < max_dx ) &&
			( dy >= min_dy ) && ( dy < max_dy ) && ( dx >= min_dx ))
		{
			char index = BDMod(dx) + plane->pelAccuracy * BDMod(dy);
			unsigned int sad = plane->SAD(blockReference,
				plane->interpolatedPreviousFrame[index] + offset + BDDiv(dx) + BDDiv(dy) * plane->pitch,
				plane->pitch, plane->pitch);
			unsigned int sum = MIN(sad + ((plane->lambda * SquareDifferenceNorm(hierarchalPredictor, dx, dy)) >> 8), sad + ((plane->lambda * SquareDifferenceNorm(medianPredictor, dx, dy)) >> 8));

			if ( sum < currentSAD ) 
			{
				currentVector.x = dx;
				currentVector.y = dy;
				currentSAD = sum;
				*dir = val;
			}
		}
	}

	/*! \brief Test of a potential motion vector.
	 *
	 *	This time, we do not take into account the direction.
	 *	\param dx X coordinate of the tested vector.
	 *	\param dy Y coordinate of the tested vector.
	 */
	___INLINE___ void CheckMV(int dx, int dy)
	{
		if (( plane->SADThreshold ) && ( dx < max_dx ) &&
			( dy >= min_dy ) && ( dy < max_dy ) && ( dx >= min_dx ))
		{
			char index = BDMod(dx) + plane->pelAccuracy * BDMod(dy);
			unsigned int sad = plane->SAD(blockReference,
				plane->interpolatedPreviousFrame[index] + offset + BDDiv(dx) + BDDiv(dy) * plane->pitch,
				plane->pitch, plane->pitch);
			unsigned int sum = MIN(sad + ((plane->lambda * SquareDifferenceNorm(hierarchalPredictor, dx, dy)) >> 8), sad + ((plane->lambda * SquareDifferenceNorm(medianPredictor, dx, dy)) >> 8));

			if ( sum < currentSAD ) 
			{
				currentVector.x = dx;
				currentVector.y = dy;
				currentSAD = sum;
			}
		}
	}

	/*! \brief Advanced Diamond search used by XviD
	 *	\param length Length ( in (1 / pelRefiningOrder)th pixel ) of the step during the research of the motion vector
	 */
	void DiamondSearch(int length);

	void ExhaustiveSearch(VECTOR mv, int length);

	void SquareSearch(VECTOR mv);

	void OneTimeSearch(VECTOR mv, int length);

	void NStepSearch(VECTOR mv, int length);

public : 

	/*! \brief Constructor of the class.
	 *	\param x_pos X coordinate of the first pixel of the block.
	 *	\param y_pos Y coordinate of the first pixel of the block.
	 *	\param w Width of the frame.
	 *	\param h Height of the frame.
	 *	\param bw Width of the block.
	 *	\param bh Height of the block.
	 *	\param pro Precision of the interpolation.
	 *	\param isMMX Flag determining the use of MMX optimizations.
	 *	\param isiSSE Flag determining the use of iSSE optimizations.
	 */
	BlockData(int x_pos, int y_pos, int w, int h, int bw, int bh, int pro, int mxd, int l, bool isMMX, bool isiSSE, PlaneOfBlocks *p);

	~BlockData();

	/*! \brief Initialization of the block
	 *
	 *	Some initializations have to be made on the first frame of the clip, that's why
	 *	they aren't done in the constructor.
	 *	\param pitch Pitch of the frame
	 *	\param intf Pointer on the interpolations of the frame
	 */
	void Initialize(int pitch,  unsigned char * const *intf);

	/*! \brief Find the best motion vector for the block
	 *	\param initMV First vector searched. The better the prediction, the better the result
	 *	and the faster the algorithm.
	 *	\param s Scale of search. For the moment, it's in fact pelRefiningOrder, but that may change
	 */
	void SearchMV(VECTOR *initMV, int externalPredictorsCount, int step, SearchType st, PredictionType pt);

	void PostProcessing(int threshold);

	//void SearchExhaustivMV(VECTOR *initMV, int s);

	/*! \brief Compute the sum of the square difference of the motion vector from the 
	 *  motion vectors of the surrounding blocks.
	 */
	void ComputeDifferenceFromNeighbours();

	/*! \brief Change the pointers to the frames.
	 *	\param s Current frame.
	 *	\param sp Previous frame.
	 */
	void SetReferences(const unsigned char *s, const unsigned char *sp);

	void FetchCurrentPredictors();

	void FetchPreviousPredictors();

	void ComputeSAD();
	void ReadVectorFromFile(MVFileHandler *input, MVVectorsStats *mvstats, int *predx, int *predy);
	void WriteVectorFromFile(MVFileHandler *output, MVVectorsStats *mvstats, int *predx, int *predy);

	// No need to comment these ones.

	___INLINE___ unsigned int GetX() const { return x; }
	___INLINE___ unsigned int GetY() const { return y; }
	//___INLINE___ unsigned int GetDifferenceFromNeighbours() const { return differenceFromNeighbours; }
	___INLINE___ VECTOR GetFinalMV() const { return finalVector; }
	___INLINE___ unsigned int GetFinalMVLength() const { return finalVectorLength; }
	___INLINE___ unsigned int GetFinalSAD() const { return finalSAD;}
	//VECTOR GetMedianPredictor();
	//unsigned int GetMedianSAD();
};

#endif
