// Create an overlay mask with the motion vectors

// See legal notice in Copying.txt for more information

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA, or visit
// http://www.gnu.org/copyleft/gpl.html .

#include "MVTracking.h"

#include <math.h>
#include <memory.h>

MVTracking::MVTracking(PClip _child, PClip mvs, int x, int y, int w, int h, const MotionParameters &params, IScriptEnvironment* env) :
GenericMotionFilter(_child, params, mvs->GetVideoInfo(), env), mvclip(mvs)
{
	CreateFGOP(&fgop);
	blocksMarked = new bool[nBlkCount];
	blocksToMark = new bool[nBlkCount];
	for ( int j = 0; j < nBlkY; j++ )
		for ( int i = 0; i < nBlkX; i++ )
			blocksToMark[i + j * nBlkX] = ( i * nBlkSize >= x ) && ( (i + 1) * nBlkSize < w + x )
								     		  && ( j * nBlkSize >= y ) && ( (j + 1) * nBlkSize < h + y );
}


MVTracking::~MVTracking()
{

}

bool MVTracking::testBlock(int x, int y)
{
	if (( x < nWidth ) && ( y < nHeight ) && ( x >= 0 ) && ( y >= 0 ))
		return blocksMarked[x / nBlkSize + (y / nBlkSize) * nBlkX];
	else
		return false;
}

bool MVTracking::wasBlockMarked(int x, int y)
{
	return testBlock(x, y) || testBlock(x + 7, y) || testBlock(x, y + 7) || testBlock(x + 7, y + 7);
}


void MVTracking::ShowAndSaveBlocks(unsigned char *dest, int dpitch)
{
	for ( int j = 0; j < nBlkY; j++ )
		for ( int i = 0; i < nBlkX; i++ )
			for ( int y = j * nBlkSize; y < j * nBlkSize + nBlkSize; y++ )
				for ( int x = i * nBlkSize; x < i * nBlkSize + nBlkSize; x++ )
					dest[y*dpitch + x] = ( blocksToMark[i + j * nBlkX] ) ? 255 : 0;

	for ( int k = 0; k < nBlkCount; k++ )
		blocksMarked[k] = blocksToMark[k];
}

PVideoFrame __stdcall MVTracking::GetFrame(int n, IScriptEnvironment* env)
{
	PVideoFrame			src			 = child->GetFrame(n, env);
	const unsigned char	*srcp_y		 = src->GetReadPtr(PLANAR_Y);
	const unsigned char	*srcp_u		 = src->GetReadPtr(PLANAR_U);
	const unsigned char	*srcp_v		 = src->GetReadPtr(PLANAR_V);
	const int			src_pitch_y  = src->GetPitch(PLANAR_Y);
	const int			src_pitch_uv = src->GetPitch(PLANAR_U);

	PVideoFrame			der			 = env->NewVideoFrame(vi);
	unsigned char		*derp_y		 = der->GetWritePtr(PLANAR_Y);
	unsigned char		*derp_u		 = der->GetWritePtr(PLANAR_U);
	unsigned char		*derp_v		 = der->GetWritePtr(PLANAR_V);
	const int			der_pitch_y	 = der->GetPitch(PLANAR_Y);
	const int			der_pitch_uv = der->GetPitch(PLANAR_U);

	GetVectorStream(n, env, mvclip, fgop);

	/*
		We test if the frame asked is the first one, because we can't calculate
		motion vectors for this one
	*/

	const FakePlaneOfBlocks &blocks = fgop->GetPlane(0);

	int mean_x = 0, mean_y = 0;
	int count = 0;
	if ( (!IsSceneChange(fgop)) && fgop->GetValidity() )
	{
		for ( int i = 0; i < nBlkCount; i++ )
		{
			int x = blocks[i].GetX() + blocks[i].GetMV().x;
			int y = blocks[i].GetY() + blocks[i].GetMV().y;
			blocksToMark[i] = wasBlockMarked(x, y);
			if ( blocksToMark[i] )
			{
				count++;
				mean_x += blocks[i].GetMV().x;
				mean_y += blocks[i].GetMV().y;
			}
		}
		mean_x = (mean_x + (count / 2)) / count;
		mean_y = (mean_y + (count / 2)) / count;
		for ( int i = 0; i < nBlkCount; i++ )
		{
			if (( abs(blocks[i].GetMV().x - mean_x ) > 2 ) ||
				( abs(blocks[i].GetMV().y - mean_y ) > 2 ))
				blocksToMark[i] = false;
		}
	}
	else {
		memset(derp_y, 0, der_pitch_y * nHeight);
	}

	ShowAndSaveBlocks(derp_y, der_pitch_y);

	memset(derp_u, 0, der_pitch_uv * nHeight / 2);
	memset(derp_v, 0, der_pitch_uv * nHeight / 2);

	return der;
}