// See legal notice in Copying.txt for more information

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA, or visit
// http://www.gnu.org/copyleft/gpl.html .

#include "Resizer.h"
#include "Interpolation.h"
#define M_PI 3.14159265358979323846

double PointResampling::f(double x)//  - added by Fizick in v0.9.10
{  
//  x = fabs(x);
  return (x<0.5 && x>=-0.5) ? 1.0 : 0.0; // simulate PointResampling with Bilinear framework
}

double BilinearResampling::f(double x)
{  
  x = fabs(x);
  return (x<1.0) ? 1.0-x : 0.0;
}

BicubicResampling::BicubicResampling (double b=1./3., double c=1./3.) 
{
  p0 = (   6. -  2.*b            ) / 6.;
  p2 = ( -18. + 12.*b +  6.*c    ) / 6.;
  p3 = (  12. -  9.*b -  6.*c    ) / 6.;
  q0 = (            8.*b + 24.*c ) / 6.;
  q1 = (         - 12.*b - 48.*c ) / 6.;
  q2 = (            6.*b + 30.*c ) / 6.;
  q3 = (      -     b -  6.*c    ) / 6.;
}

double BicubicResampling::f (double x)
{
    x = fabs(x);
    return (x<1) ? (p0+x*x*(p2+x*p3)) : (x<2) ? (q0+x*(q1+x*(q2+x*q3))) : 0.0;
}

double LanczosResampling::sinc(double value)
{
	if (value != 0.0)
	{
		value *= M_PI;
		return sin(value) / value;
	}
	else
	{
		return 1.0;
	}
}

double LanczosResampling::f(double value)
{
	if (value < 0.0)
	{
		value = -value;
	}
	if (value < 3.0)
	{
		return (sinc(value) * sinc(value / 3.0));
	}
	else
	{
		return 0.0;
	}
}


Upsizer::Upsizer(int w, int h, int wr, int hr, int ua, int uo,
				 int offx, int offy, ResamplingFunction *rf)
{
	width = w;
	height = h;
	widthRatio = wr;
	heightRatio = hr;
	widthCoefficients = new int*[widthRatio];
	heightCoefficients = new int*[heightRatio];
	resamplingFunction = rf;
	intScale = 16384;
	logScale = 14;

	widthLength = 2 * resamplingFunction->support() + 1;
	widthHalfLength = resamplingFunction->support();
	heightLength = 2 * resamplingFunction->support() + 1;
	heightHalfLength = resamplingFunction->support();

	xOrigin = ua;
	yOrigin = uo;
	xOffset = offx;
	yOffset = offy;

	destination = new unsigned char *[widthRatio * heightRatio];

	BuildCoefficients();
}

Upsizer::~Upsizer()
{
	int i;
	delete resamplingFunction;
	for ( i = 0; i < widthRatio; i++ )
		delete[] widthCoefficients[i];
	delete widthCoefficients;
	for ( i = 0; i < heightRatio; i++ )
		delete[] heightCoefficients[i];
	delete heightCoefficients;

	delete[] destination;
}

void Upsizer::BuildCoefficients()
{
	int i, j;
	for ( i = 0 ; i < widthRatio; i++ )
	{
		double offset = ((double)(i - xOrigin)) / ((double)widthRatio);
		double total = 0.0;
	
		widthCoefficients[i] = new int[widthLength];

		for ( j = 0; j < widthLength; j++ )
			total += resamplingFunction->f(offset + j - widthHalfLength);

		double total2 = 0.0;
		for ( j = 0; j < widthLength; j++ )
		{
			double total3 = total2 + resamplingFunction->f(offset + j - widthHalfLength) / total;
			widthCoefficients[i][widthLength - j - 1] = int(total3 * intScale + 0.5) - int(total2 * intScale + 0.5);
			total2 = total3;
		}
	}

	for ( i = 0 ; i < heightRatio; i++ )
	{
		double offset = ((double)(i - yOrigin)) / ((double)heightRatio);
		double total = 0.0;

		heightCoefficients[i] = new int[heightLength];

		for ( j = 0; j < heightLength; j++ )
			total += resamplingFunction->f(offset + j - heightHalfLength);

		double total2 = 0.0;
		for ( j = 0; j < heightLength; j++ )
		{
			double total3 = total2 + resamplingFunction->f(offset + j - heightHalfLength) / total;
			heightCoefficients[i][heightLength - j - 1] = int(total3 * intScale + 0.5) - int(total2 * intScale + 0.5);
			total2 = total3;
		}
	}
}

void Upsizer::Resize(const unsigned char *source, unsigned char **dest,
					 int srcPitch, int destPitch, bool doOriginal)
{
	// Initialization

    int i, j, k, m, h;
	int offScale = intScale / 2;
	int offset = xOffset + yOffset * destPitch;
	for ( j = 0; j < heightRatio; j++ )
		for ( i = 0; i < widthRatio; i++ )
			destination[i + j * widthRatio] = dest[i + j * widthRatio] + offset;

	for ( i = 0; i < heightRatio; i++ )
	{
		if ( i == yOrigin )
		{
			if ( doOriginal )
			{
				unsigned char *dest = destination[xOrigin + i * widthRatio];
				const unsigned char *src = source;
				for ( j = 0; j < height; j++ )
				{
					for ( k = 0; k < width; k++ )
						dest[k] = src[k];
					dest += destPitch;
					src += srcPitch;
				}
			}
		}
		else {
			const int *hc = heightCoefficients[i];
			for ( j = 0; j < width; j++ )
			{
				unsigned char *dest = destination[xOrigin + i * widthRatio] + j;
				const unsigned char *src = source + j;
				for ( k = 0; k < heightHalfLength; k++ )
				{
					int result = 0;
					int normalization = 0;
					for ( m = 0; m < k + heightHalfLength; m++ )
					{
						result += hc[m + heightHalfLength - k] * src[m * srcPitch];
						normalization += hc[m + heightHalfLength - k];
					}
					int offset = normalization / 2;
					dest[k * destPitch] = FitIntoUCDiv(result + offset, normalization);
				}

				for ( k = 0; k < height - 2 * heightHalfLength; k++ )
				{
					int result = 0;
					for ( m = 0; m < heightLength; m++ )
					{
						result += hc[m] * src[(m + k) * srcPitch];
					}
					dest[(k + heightHalfLength) * destPitch] = FitIntoUCOff(result + offScale, logScale);
				}

				for ( k = 0; k < heightHalfLength; k++ )
				{
					int result = 0;
					int normalization = 0;
					for ( m = 0; m < heightLength - k - 1; m++ )
					{
						result += hc[m] * src[(m + height + k + 1 - heightLength) * srcPitch];
						normalization += hc[m];
					}
					int offset = normalization / 2;
					dest[(height - heightHalfLength + k) * destPitch] = FitIntoUCDiv(result + offset, normalization);
				}
			}
		}
	}

	for ( h = 0; h < widthRatio; h++ )
	{
		if ( h != xOrigin )
		{
			const int *wc = widthCoefficients[h];
			for ( i = 0; i < heightRatio; i++ )
			{
				const unsigned char *src = destination[xOrigin + widthRatio * i];
				unsigned char *dest = destination[h + widthRatio * i];
				
				for ( j = 0; j < height; j++ )
				{
					for ( k = 0; k < widthHalfLength; k++ )
					{
						int result = 0;
						int normalization = 0;
						for ( m = 0; m < k + widthLength - widthHalfLength; m++ )
						{
							result += wc[m + widthHalfLength - k] * src[m];
							normalization += wc[m + widthHalfLength - k];
						}
						int offset = normalization / 2;
						dest[k] = FitIntoUCDiv(result + offset, normalization);
					}

					for ( k = 0; k < width - 2 * widthHalfLength; k++ )
					{
						int result = 0;
						for ( m = 0; m < widthLength; m++ )
						{
							result += wc[m] * src[m + k];
						}
						dest[k + widthHalfLength] = FitIntoUCOff(result + offScale, logScale);
					}

					for ( k = 0; k < widthHalfLength; k++ )
					{
						int result = 0;
						int normalization = 0;
						for ( m = 0; m < widthLength - k - 1; m++ )
						{
							result += wc[m] * src[m + width + k + 1 - widthLength];
							normalization += wc[m];
						}
						int offset = normalization / 2;
						dest[width - widthHalfLength + k] = FitIntoUCDiv(result + offset, normalization);
					}
					dest += destPitch;
					src += destPitch;
				}
			}
		}
	}
}
