#ifndef __PORTABILITY__
#define __PORTABILITY__

#pragma warning(disable:4103) // disable pack to change alignment warning ( stlport related )
#pragma warning(disable:4800) // disable warning about bool to int unefficient conversion

#include <stdio.h>
#include <stdarg.h>
#include "avisynth.h"

#define MOTION_DEBUG          // allows to output debug information to the debug output
#define MOTION_PROFILE        // allows to make a profiling of the motion estimation


#define MOTION_MAGIC_KEY 31415926

struct VECTOR {
	int x;
	int y;
   int sad;
};

typedef unsigned char Uint8;

enum MVPlaneSet {
   YPLANE = 1,
   UPLANE = 2,
   VPLANE = 4,
   YUPLANES = 3,
   YVPLANES = 5,
   UVPLANES = 6,
   YUVPLANES = 7
};

/*! \brief Search type : defines the algorithm used for minimizing the SAD */
enum SearchType {
	EXHAUSTIVE = 1,
	LOGARITHMIC = 2,
	ONETIME = 4,
	NSTEP = 8,
	SQUARE = 16
};

/*! \brief Macros for accessing easily frame pointers & pitch */
#define YRPLAN(a) (a)->GetReadPtr(PLANAR_Y)
#define YWPLAN(a) (a)->GetWritePtr(PLANAR_Y)
#define URPLAN(a) (a)->GetReadPtr(PLANAR_U)
#define UWPLAN(a) (a)->GetWritePtr(PLANAR_U)
#define VRPLAN(a) (a)->GetReadPtr(PLANAR_V)
#define VWPLAN(a) (a)->GetWritePtr(PLANAR_V)
#define YPITCH(a) (a)->GetPitch(PLANAR_Y)
#define UPITCH(a) (a)->GetPitch(PLANAR_U)
#define VPITCH(a) (a)->GetPitch(PLANAR_V)

/*! \brief usefull macros */
#define MAX(a,b) (((a) < (b)) ? (b) : (a))
#define MIN(a,b) (((a) > (b)) ? (b) : (a))


#define MOTION_CALC_SRC_LUMA        0x00000001
#define MOTION_CALC_REF_LUMA        0x00000002
#define MOTION_CALC_VAR             0x00000004
#define MOTION_CALC_BLOCK           0x00000008
#define MOTION_USE_MMX              0x00000010
#define MOTION_USE_ISSE             0x00000020
#define MOTION_IS_BACKWARD          0x00000040
#define MOTION_SMALLEST_PLANE       0x00000080
#define MOTION_COMPENSATE_LUMA      0x00000100
#define MOTION_COMPENSATE_CHROMA_U  0x00000200
#define MOTION_COMPENSATE_CHROMA_V  0x00000400

static const VECTOR zeroMV = { 0, 0, -1 };

#define inline inline

/*! \brief debug printing information */
#ifdef MOTION_DEBUG
static inline void DebugPrintf(char *fmt, ...)
{
   va_list args;
   char buf[1024];

   va_start(args, fmt);
   vsprintf(buf, fmt, args);
   OutputDebugString(buf);
}
#else
static inline void DebugPrintf(char *fmt, ...)
{ }
#endif

// profiling
#define MOTION_PROFILE_ME              0
#define MOTION_PROFILE_LUMA_VAR        1
#define MOTION_PROFILE_COMPENSATION    2
#define MOTION_PROFILE_INTERPOLATION   3
#define MOTION_PROFILE_COUNT           4

#ifdef MOTION_PROFILE

extern int ProfTimeTable[MOTION_PROFILE_COUNT];
extern int ProfResults[MOTION_PROFILE_COUNT * 2];
extern __int64 ProfCumulatedResults[MOTION_PROFILE_COUNT * 2];

#define PROFILE_START(x) \
{ \
   __asm { rdtsc }; \
   __asm { mov [ProfTimeTable + 4 * x], eax }; \
}

#define PROFILE_STOP(x) \
{ \
   __asm { rdtsc }; \
   __asm { sub eax, [ProfTimeTable + 4 * x] }; \
   __asm { add eax, [ProfResults + 8 * x] }; \
   __asm { mov [ProfResults + 8 * x], eax }; \
   ProfResults[2 * x + 1]++; \
}

inline static void PROFILE_INIT()
{
   for ( int i = 0; i < MOTION_PROFILE_COUNT; i++ )
   {
      ProfTimeTable[i]                 = 0;
      ProfResults[2 * i]               = 0;
      ProfResults[2 * i + 1]           = 0;
      ProfCumulatedResults[2 * i]      = 0;
      ProfCumulatedResults[2 * i + 1]  = 0;
   }
}
inline static void PROFILE_CUMULATE()
{
   for ( int i = 0; i < MOTION_PROFILE_COUNT; i++ )
   {
      ProfCumulatedResults[2 * i]     += ProfResults[2 * i];
      ProfCumulatedResults[2 * i + 1] += ProfResults[2 * i + 1];
      ProfResults[2 * i]     = 0;
      ProfResults[2 * i + 1] = 0;
   }
}
inline static void PROFILE_SHOW()
{
   __int64 nTotal = 0;
   for ( int i = 0; i < MOTION_PROFILE_COUNT; i++ )
      nTotal += ProfCumulatedResults[2*i];

   DebugPrintf("ME            : %2.1f %%", (double)ProfCumulatedResults[2 * MOTION_PROFILE_ME] * 100.0f / (nTotal + 1));
   DebugPrintf("LUMA & VAR    : %2.1f %%", (double)ProfCumulatedResults[2 * MOTION_PROFILE_LUMA_VAR] * 100.0f / (nTotal + 1));
   DebugPrintf("COMPENSATION  : %2.1f %%", (double)ProfCumulatedResults[2 * MOTION_PROFILE_COMPENSATION] * 100.0f / (nTotal + 1));
   DebugPrintf("INTERPOLATION : %2.1f %%", (double)ProfCumulatedResults[2 * MOTION_PROFILE_INTERPOLATION] * 100.0f / (nTotal + 1));
}
#else
#define PROFILE_START(x)
#define PROFILE_STOP(x)
#define PROFILE_INIT()
#define PROFILE_CUMULATE()
#define PROFILE_SHOW()
#endif

#endif

